<?php

namespace App\Http\Controllers;

use DB;
use App\City;
use App\Traits\ApiResponser;
use Illuminate\Http\Request;
use \App\Services\City\CityServices;
use App\Http\Requests\City\CityRequest;
use App\Http\Middleware\PermissionsMiddleware;

class CityController extends Controller
{
    use ApiResponser;

    public $service;

    /**
     * constructor
     */
	function __construct()
	{
        $this->service = new CityServices();
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('city-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('city-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('city-delete'))->handle($request, $next);
        })->only(['destroy']);
	}

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //
        try {
            $page = $request->page ? $request->page : 1;
            $search = $request->search ? $request->search : '';
            $perPage = $request->perPage ? $request->perPage : 12;
            // retrun data
            return [
                'success' => true,
                'message' => 'Listado de ciudades.',
                'data' => $this->service->getCity($page, $perPage, $search),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(CityRequest $request)
    {
        //
        DB::beginTransaction();
        try {
            $data = $this->service->saveCity($request->all());
            DB::commit();
            return [
                'success' => true,
                'message' => 'Se ha guardado la ciudad correctamente.',
                'data' => $data,
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\City  $city
     * @return \Illuminate\Http\Response
     */
    public function show(City $city)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\City  $city
     * @return \Illuminate\Http\Response
     */
    public function edit(City $city)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\City  $city
     * @return \Illuminate\Http\Response
     */
    public function update(CityRequest $request, City $city)
    {
        //
        DB::beginTransaction();
        try {
            $data = $this->service->updateCity($city, $request->all());
            DB::commit();
            return [
                'success' => true,
                'message' => 'Se ha modificado la ciudad correctamente.',
                'data' => $data,
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\City  $city
     * @return \Illuminate\Http\Response
     */
    public function destroy(City $city)
    {
        //
        DB::beginTransaction();
        try {
            $data = $this->service->deleteCity($city->id);
            DB::commit();
            return [
                'success' => true,
                'message' => 'Se ha eliminado la ciudad correctamente.',
                'data' => $data
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * export cities to excell
     */
    public function exportExcel() {
        try {
            return $this->service->exportExcel();
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function listCities(Request $request)
    {
        //
        try {
            $page = $request->page ? $request->page : 1;
            $search = $request->search ? $request->search : '';
            $perPage = $request->perPage ? $request->perPage : 1000;
            // retrun data
            return [
                'success' => true,
                'message' => 'Listado de ciudades.',
                'data' => $this->service->getCity($page, $perPage, $search),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }
}
