<?php

namespace App\Http\Controllers;

set_time_limit(300000000);

use App\DispatchRelation;
use Illuminate\Http\Request;
use DB;
use App\Traits\ApiResponser;
use App\Http\Requests\DispatchRelation\Store;
use PDF;
use App\DispatchRelationItem;
use Carbon\Carbon;
use App\Services\Relations\RelationDispathService;
use App\Http\Middleware\PermissionsMiddleware;

class DispatchRelationController extends Controller
{
	use ApiResponser;

	public $service;

	/**
	 * constructor
	 */
	public function __construct()
	{
		$this->service = new RelationDispathService();
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('relation-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('relation-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('relation-delete'))->handle($request, $next);
        })->only(['destroy']);
	}

	/**
	 * Display a listing of the resource.
	 *
	 * @return \Illuminate\Http\Response
	 */
	public function index(Request $request)
	{
		try {
            // get query params
            $perPage = $request->perPage ? $request->perPage : 10;
            $from = $request->from ? $request->from : Carbon::now()->startOfMonth();
            $to = $request->to ? $request->to . ' 23:59:59' : Carbon::now()->endOfMonth();
            $search = $request->search ? $request->search : '';
            // get query
			$dispatchRelations = $this->service->listRelation($perPage, $from, $to, $search);
			// return data
            return $this->successResponse([
				'success' => true,
				'data' => $dispatchRelations
			]);
		} catch (\Exception $e) {
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * Show the form for creating a new resource.
	 *
	 * @return \Illuminate\Http\Response
	 */
	public function create()
	{
		//
	}

	/**
	 * Store a newly created resource in storage.
	 *
	 * @param  \Illuminate\Http\Request  $request
	 * @return \Illuminate\Http\Response
	 */
	public function store(Store $request)
	{
		//
		DB::begintransaction();
		try {
			$dispatchRelation = DispatchRelation::create([
				'status' => 'En curso',
				'domiciliario_id' => $request->domiciliario_id
			]);
			$domicilioIds = [];
			// validamos los items
			foreach ($request->items as $key => $item) {
				$existingItem = DispatchRelationItem::where('domicilio_id', $item['domicilio_id'])->first();
				if ($existingItem) {
					$existingDispatchRelation = DispatchRelationItem::where('dispatch_relation_id', $existingItem->dispatch_relation_id)->count();
					if ($existingDispatchRelation>=2) {
						DispatchRelationItem::where('domicilio_id', $item['domicilio_id'])
						->where('dispatch_relation_id', $existingItem->dispatch_relation_id)
						->where('id', $existingItem->id)
						->delete();
					} else {
						$existingDispatchRelation = DispatchRelation::where('id', $existingItem->dispatch_relation_id)->first();
						if ($existingDispatchRelation) {
							$deleteItems = DispatchRelationItem::where('domicilio_id', $item['domicilio_id'])->delete();
							$existingDispatchRelation->delete();
						} else {
							$deleteItems = DispatchRelationItem::where('domicilio_id', $item['domicilio_id'])->delete();
						}
					}
				}

				DispatchRelationItem::create([
					'domicilio_id' => $item['domicilio_id'],
					'dispatch_relation_id' => $dispatchRelation->id,
				]);
				array_push($domicilioIds, $item['domicilio_id']);
			}
			// asignamos el domiciliario de la relacion a los pedidos que vienen
			$updateDomicilio = \App\Domicilio::whereIn('id', $request->items)->update(['domiciliario_id' => $request->domiciliario_id]);
			$updateDomicilioStatus = \App\Domicilio::whereIn('id', $request->items)->where('status', 'Reprogramado')->update(['status' => 'Pendiente']);
			// retornamos la data
			DB::commit();
			return $this->successCreatedResponse([
				'data' => $dispatchRelation,
				'message' => 'Relación de despacho creada correctamente.',
				'pdf' => $this->laodPdf($dispatchRelation)
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * Display the specified resource.
	 *
	 * @param  \App\DispatchRelation  $dispatchRelation
	 * @return \Illuminate\Http\Response
	 */
	public function show(DispatchRelation $dispatchRelation)
	{
		//
		try {
			return $dispatchRelation;
		} catch (\Exception $e) {
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * Show the form for editing the specified resource.
	 *
	 * @param  \App\DispatchRelation  $dispatchRelation
	 * @return \Illuminate\Http\Response
	 */
	public function edit(DispatchRelation $dispatchRelation)
	{
		//
	}

	/**
	 * Update the specified resource in storage.
	 *
	 * @param  \Illuminate\Http\Request  $request
	 * @param  \App\DispatchRelation  $dispatchRelation
	 * @return \Illuminate\Http\Response
	 */
	public function update(Request $request, DispatchRelation $dispatchRelation)
	{
		//
		DB::begintransaction();
		try {
			$dispatchRelation->update(['status' => 'Completada']);
			DB::commit();
			return $this->successResponse([
				'success' => true,
				'message' => 'Se ha completado la relación de despacho',
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * Remove the specified resource from storage.
	 *
	 * @param  \App\DispatchRelation  $dispatchRelation
	 * @return \Illuminate\Http\Response
	 */
	public function destroy(DispatchRelation $dispatchRelation)
	{
		//
		DB::begintransaction();
		try {
			$dispatchRelation->delete();
			DB::commit();
			return $this->successResponse([
				'success' => true,
				'message' => 'Se ha eliminado la relación de despacho',
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * print pdf
	 *
	 */
	public function emitPdf(Request $request)
	{
		try {
			if (!$request->has('id'))
				return $this->errorResponse('Debes enviar el id de la relación a generar.', 400);
			$relation = DispatchRelation::findOrFail($request->id);
            $configuration = \App\Configuration::all()->first();
			// Generar el PDF a partir de la vista
			$pdf = PDF::loadView('onx.relation', compact('relation', 'configuration'));
			// Descargar el archivo PDF
			$path = public_path('pdf/');
			$fileName =  'relation_' . $relation->id . '.pdf';
			$pdf->save($path . '/' . $fileName);
			$file = public_path() . "/pdf/" . $fileName;
			if (ENV('APP_ENV') == 'production') {
                $url = url('pdf');
            } else {
                $url = url('pdf');
            }
			return "{$url}/{$fileName}";
		} catch (\Exception $e) {
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * print pdf
	 *
	 */
	public function laodPdf(DispatchRelation $relation)
	{
		try {
			// Generar el PDF a partir de la vista
            $configuration = \App\Configuration::all()->first();
			$pdf = PDF::loadView('onx.relation', compact('relation', 'configuration'));
			// Descargar el archivo PDF
			$path = public_path('pdf/');
			$fileName =  'relation_' . $relation->id . '.pdf';
			$pdf->save($path . '/' . $fileName);
			$file = public_path() . "/pdf/" . $fileName;
			if (ENV('APP_ENV') == 'production') {
                $url = url('pdf');
            } else {
                $url = url('pdf');
            }
			return "{$url}/{$fileName}";
		} catch (\Exception $e) {
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

	/**
	 * List relation by domiciliario
	 *
	 * @return \Illuminate\Http\Response
	 */
	public function getTodayRelation(Request $request)
	{
		//
		try {
			$start = Carbon::now()->format('Y-m-d') . ' 00:00:00';
			$end = Carbon::now()->format('Y-m-d') . ' 23:59:59';
			$dispatchRelations = DispatchRelation::select(
				'dispatch_relations.*',
				//'log_senders.name as sender_name',
				'domiciliarios.name as domiciliario_name',
				// DB::raw("COUNT(dispatch_relation_items.id) as totalItems")
			)
				//->leftJoin('log_senders', 'log_senders.id', '=', 'dispatch_relations.log_sender_id')
				->leftJoin('domiciliarios', 'domiciliarios.id', '=', 'dispatch_relations.domiciliario_id')
				->leftJoin('dispatch_relation_items', 'dispatch_relation_items.dispatch_relation_id', '=', 'dispatch_relations.id')
				->where('dispatch_relations.domiciliario_id', $request->domiciliarioId)
				->whereBetween('dispatch_relations.created_at', [$start, $end])
				->groupBy('dispatch_relations.id')
				->get();

			// Recorrer cada relación de despacho para calcular estadísticas - HERC
			foreach ($dispatchRelations as $dispatchRelation) {
				// Calcular el total de domicilios realizados. - HERC
				$dispatchRelation->totalItems = DB::table('dispatch_relation_items')
					->where('dispatch_relation_id', $dispatchRelation->id)
					->count();

				// Calcular la cantidad domicilios entregados. - HERC
				$dispatchRelation->delivered = DB::table('dispatch_relation_items')
					->join('domicilios', 'dispatch_relation_items.domicilio_id', '=', 'domicilios.id')
					->where('dispatch_relation_id', $dispatchRelation->id)
					->where('domicilios.status', 'Entregado')
					->count();

				// Obtener el valor del campo "total" de la tabla "domicilios" y se usa para mostrar el recaudo total. - HERC
				$dispatchRelation->recaudo = DB::table('domicilios')
					->join('dispatch_relation_items', 'domicilios.id', '=', 'dispatch_relation_items.domicilio_id')
					->where('dispatch_relation_items.dispatch_relation_id', $dispatchRelation->id)
					->sum('domicilios.total');
			}

			return $this->successResponse([
				'success' => true,
				'data' => $dispatchRelations
			]);
		} catch (\Exception $e) {
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

    /**
     * List relations by domiciliario and do some filters
     * @param Request $request
     */
    public function listByDomiciliario(Request $request) {
        try {
            // get filter data
            $start = $request->from ? $request->from : Carbon::now()->startOfWeek()->format('Y-m-d');
            $end = $request->to ? $request->to : Carbon::now()->endOfWeek()->format('Y-m-d');
            $page = $request->page ? $request->page : 1;
            $perPage = $request->perPage ? $request->perPage : 12;
            $domiciliarioId = $request->domiciliarioId;
            // do query and return data
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getRelationList(
                    $start . ' 00:00:00',
                    $end . ' 23:59:59',
                    $page,
                    $perPage,
                    $domiciliarioId
                )
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * List order by relation dispatch
     * @param Request $request
     */
    public function loadOrderByRelationOrNot(Request $request) {
        try {
            $start = $request->from ? $request->from : Carbon::now()->startOfMonth()->format('Y-m-d');
            $end = $request->to ? $request->to : Carbon::now()->endOfMonth()->format('Y-m-d');
            $page = $request->page ? $request->page : 1;
            $perPage = $request->perPage ? $request->perPage : 12;
            $relationId = $request->relationId ? $request->relationId : null;
            $domiciliarioId = $request->domiciliarioId;
            $status = $request->status ? $request->status : 'todos';
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getOrderByRelationOrNot(
                    $start . ' 00:00:00',
                    $end . ' 23:59:59',
                    $page,
                    $perPage,
                    $relationId,
                    $domiciliarioId,
                    $status
                )
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * Filter order by id
     * @param $id
     * @param Request $request
     */
    public function getOrderById($id, Request $request) {
        try {
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->getOrderById($id)
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
	 * mark as compelte the relation dispatch
	 */
	public function completeRelation($id)
	{
		//
		DB::begintransaction();
		try {
			$relation = $this->service->completeRelation($id);
			DB::commit();
			return $this->successResponse([
				'success' => true,
				'message' => 'Se ha completado la relación de despacho',
                'data' => $relation,
			]);
		} catch (\Exception $e) {
			DB::rollback();
			return $this->handlerException($e->getMessage(), 400);
		}
	}
}
