<?php

namespace App\Http\Controllers;

use App\Domiciliario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\User;
use App\Helpers\Helper;
use Response;
use App\Http\Controllers\AuditController;
use App\Http\Resources\Domiciliario\DomiciliarioResource;
use App\Http\Resources\Domiciliario\DomiciliarioCollection;
use Illuminate\Support\Facades\Event;
use App\Events\Domiciliario\DomiciliarioCreate;
use App\Events\Domiciliario\DomiciliarioUpdate;
use App\Http\Requests\Domiciliario\PostRequest;
use App\Http\Requests\Domiciliario\PutRequest;
use App\Http\Controllers\AuthController;
use App\Traits\ApiResponser;
use DB;
use Illuminate\Validation\Rule; // HERC
use Illuminate\Support\Facades\Validator; // HERC
use App\Http\Middleware\PermissionsMiddleware;

class DomiciliarioController extends Controller
{
    use ApiResponser;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-update'))->handle($request, $next);
        })->only(['update']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('domiciliary-delete'))->handle($request, $next);
        })->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            // saco el id principal de la tabla recursiva usuarios.
            $domiciliarios = Domiciliario::all();
            return Helper::response('success', new DomiciliarioCollection($domiciliarios), 200);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

	// Raul
    public function search(Request $request){
        try {
            $domiciliarios = DB::table('domiciliarios')
                ->where('name', 'like', "%{$request->name}%")->get();
            return Helper::response('success', $domiciliarios, 200);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(PostRequest $request)
    {
        DB::beginTransaction();
        try {
            $mainId = $request->user()->getIdMain($request->user());

            // Validaciones - HERC
            $validator = Validator::make($request->all(), [
                 'name' => [
                    'required',
                    'min:3',
                    'max:50',
                    'regex:/^[A-Za-zÁÉÍÓÚÜáéíóúüñÑ\s]+$/',
                ],
                'email' => [
                    'required',
                    'email',
                    Rule::unique('domiciliarios', 'email'),
                ],
                'dni' => [
                    'required',
                    'max:15',
                    'min:8',
                    Rule::unique('domiciliarios', 'dni'),
                ],
                'phone' => [
                    'required',
                    'max:15'
                ],
                'password' => [
                    'required',
                    'min:8',
                    'max:12',
                    'regex:/(?=.*\d)(?=.*[a-z])(?=.*[A-Z])/',
                    'not_regex:/(.)\1{2,}/',
                ],
                'hiring_method' => [
                    'required'
                ],
            ], [
                'name.required' => 'Campo nombre requerido.',
                'name.min' => 'El nombre no debe tener menos de 3 caracteres.',
                'name.max' => 'El nombre no debe tener más de 50 caracteres.',
                'name.regex' => 'El nombre debe contener solo letras y espacios.',
                'email.required' => 'El campo correo electrónico es obligatorio.',
                'email.email' => 'Correo electrónico no válido.',
                'email.unique' => 'El correo electrónico ingresado ya está en uso.',
                'dni.required' => 'El campo DNI es obligatorio.',
                'dni.unique' => 'El DNI ingresado ya está en uso.',
                'dni.max' => 'El DNI no debe tener más de 15 caracteres.',
                'phone.required' => 'Campo teléfono requerido.',
                'phone.max' => 'Número de teléfono no válido. Debe tener maximo 15 dígitos.',
                'password.required' => 'Campo contraseña requerido.',
                'password.min' => 'La contraseña no debe tener menos de 8 caracteres.',
                'password.max' => 'La contraseña no debe tener más de 12 caracteres.',
                'password.regex' => 'La contraseña debe contener al menos una letra mayúscula, una letra minúscula y un número.',
                'password.not_regex' => 'La contraseña no debe contener tres caracteres repetidos consecutivos.',
                'hiring_method.required' => 'Campo método de contratación requerido.',
            ]);

            // Verificar si hay errores de validación - HERC
            if ($validator->fails()) {
                return $this->errorUnprocessableEntityResponse($validator->errors());
            }
            // test
            $isset = \App\User::where('email', $request->email)->first();
            if (isset($isset)) {
                return $this->handlerException('Existe un usuario con este correo electrónico');
            }

            // subimos la imagen del domiciliario
            if ($request->has('file'))
                $path = Helper::uploadImage($request->file, 'domiciliario');
            $domiciliario = Domiciliario::create(array_merge($request->all(), ['role_id' => 5]));
            $domiciliario->email = $domiciliario->email; // HERC
            $domiciliario->password = $request['password']; // HERC
            $domiciliario->username = $domiciliario->dni;
            $domiciliario->hiring_method = $request['hiring_method'];
            // le guardamos el usuario al domiciliario
            $user = AuthController::register($domiciliario->email, $domiciliario->username, $request['password'], $domiciliario->id);
            //set data to auditoria
            AuditController::store($request->user()->name, 'Ha creado al domiciliario: ' .$domiciliario->name, 'Domiciliario', $mainId);
            DB::commit();
            return Helper::response('success', ['message' => 'Se ha creado el domiciliario correctamente.',
            'data' => new DomiciliarioResource($domiciliario)], 201);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Domiciliario  $domiciliario
     * @return \Illuminate\Http\Response
     */
    public function show(Domiciliario $domiciliario)
    {
        //
        try {
            return $this->successResponse(new DomiciliarioResource($domiciliario));
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Domiciliario  $domiciliario
     * @return \Illuminate\Http\Response
     */
    public function edit(Domiciliario $domiciliario)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Domiciliario  $domiciliario
     * @return \Illuminate\Http\Response
     */
    public function update(PutRequest $request, Domiciliario $domiciliario)
    {
        try {
            $mainId = $request->user()->getIdMain($request->user());
            $user = \App\User::where('domiciliario_id', $domiciliario->id)->first();

            // Validaciones - HERC
            $validator = Validator::make($request->all(), [
                'name' => [
                    'required',
                    'min:3',
                    'max:50',
                    'regex:/^[A-Za-zÁÉÍÓÚÜáéíóúüñÑ\s]+$/',
                ],
                'email' => [
                    'required',
                    'email',
                    Rule::unique('domiciliarios', 'email')->ignore($domiciliario->id),
                    Rule::unique('users', 'email')->ignore($user->id),
                ],
                'dni' => [
                    'required',
                    'max:15',
                    'min:8',
                    Rule::unique('domiciliarios', 'dni')
                    ->ignore($domiciliario->id, 'id'),
                ],
                'phone' => [
                    'required',
                    'max:15',
                ],
                'hiring_method' => [
                    'required',
                ],
            ], [
                'name.required' => 'Campo nombre requerido.',
                'name.min' => 'El nombre no debe tener menos de 3 caracteres.',
                'name.max' => 'El nombre no debe tener más de 50 caracteres.',
                'name.regex' => 'El nombre debe contener solo letras y espacios.',
                'email.required' => 'El campo correo electrónico es obligatorio.',
                'email.email' => 'Correo electrónico no válido.',
                'email.unique' => 'El correo electrónico ingresado ya está en uso.',
                'dni.required' => 'El campo DNI es obligatorio.',
                'dni.unique' => 'El DNI ingresado ya está en uso.',
                'dni.max' => 'El DNI no debe tener más de 10 caracteres.',
                'dni.regex' => 'DNI no válido. Nomenclatura inválida.',
                'phone.required' => 'Campo teléfono requerido.',
                'phone.regex' => 'Número de teléfono no válido. Debe tener maximo 15 dígitos.',
                'password.min' => 'La contraseña no debe tener menos de 8 caracteres.',
                'password.max' => 'La contraseña no debe tener más de 12 caracteres.',
                'password.regex' => 'La contraseña debe contener al menos una letra mayúscula, una letra minúscula y un número.',
                'password.not_regex' => 'La contraseña no debe contener tres caracteres repetidos consecutivos.',
                'hiring_method.required' => 'Campo método de contratación requerido.',
            ]);

            // Verificar si la contraseña se proporciona y no está vacía en el contexto de edición
            if ($request->input('password') !== null && $request->input('password') !== '') {
                $validator->addRules([
                    'password' => [
                        'required',
                        'min:8',
                        'max:12',
                        'regex:/(?=.*\d)(?=.*[a-z])(?=.*[A-Z])/',
                        'not_regex:/(.)\1{2,}/',
                    ]
                ]);
            }

            // Verificar si hay errores de validación - HERC
            if ($validator->fails()) {
                return $this->errorUnprocessableEntityResponse($validator->errors());
            }
            // Encriptar la contraseña antes de actualizarla - HERC
            if ($request->has('password')) {
                $request['password'] = bcrypt($request['password']);
            }
            $domiciliario->update($request->all());
            $user = User::where('domiciliario_id', $domiciliario->id)
            ->first();
            if ($user) {
                $user->update([
                    'email' => $domiciliario->email, // HERC
                    'name' => $domiciliario->dni
                ]);
                if (isset($request->password)) {
                    $user->update(['password' => $request->password]); // HERC
                }
            } else {
                $domiciliario->email = $domiciliario->email; // HERC
                $domiciliario->password = $domiciliario->password; // HERC
                $domiciliario->username = $domiciliario->dni;
                $user = AuthController::register($domiciliario->email, $domiciliario->username, $domiciliario->password, $domiciliario->id);
            }
            AuditController::store($request->user()->name, 'Ha modificado al domiciliario: ' .$domiciliario->name, 'Domiciliario', $mainId);
            return Helper::response('success', ['message' => 'Se ha modificado el domiciliario correctamente', 'data' => new DomiciliarioResource($domiciliario)], 201);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Domiciliario  $domiciliario
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Domiciliario $domiciliario)
    {
        try {
            $mainId = $request->user()->getIdMain($request->user());
            AuditController::store($request->user()->name, 'Ha eliminado al domiciliario: ' .$domiciliario->name, 'Domiciliario', $mainId);
            $domiciliario->delete();
            //delete user from bbdd
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            DB::table('users')->where('domiciliario_id', '=', $domiciliario->id)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            return Helper::response('success', ['message' => 'Se ha eliminado el domiciliario correctamente', 'data' => new DomiciliarioResource($domiciliario)], 201);
        } catch (\Exception $e) {
            return $this->errorUnprocessableEntityResponse($e->getMessage());
        }
    }
}
