<?php

namespace App\Http\Controllers;

use DB;
use App\User;
use Response;
use Carbon\Carbon;
use App\Domicilio;
use App\DomicilioProduct;
use App\Http\Requests\Domicilio\BulkStatus;
use Illuminate\Http\Request;
use App\Helpers\Helper;
use App\Http\Controllers\ClientController;
use App\Http\Requests\Domicilio\DomicilioRequest;
use App\Http\Resources\Domicilio\DomicilioCollection;
use App\Http\Resources\Domicilio\DomicilioResource;
use Illuminate\Support\Facades\Event;
use App\Events\Domicilio\MainContability;
use App\Services\Domicilio\DomicilioServices;
use App\Traits\ApiResponser;
use App\Services\User\UserServices;
use Illuminate\Support\Facades\Schema;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Requests\Domicilio\OrderCreate;
use App\Http\Requests\Domicilio\UploadEvidence;
use App\Http\Requests\Domicilio\ReferenceRequest;
use App\Http\Middleware\PermissionsMiddleware;
use App\Http\Requests\Domicilio\DomicilioImportRequest;

class DomicilioController extends Controller
{
    use ApiResponser;
    public $userService, $domicilioService;

    public function __construct()
    {
        $this->userService = new UserServices();
        $this->domicilioService = new DomicilioServices();
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('last-order'))->handle($request, $next);
        })->only(['getLastOrders']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('list-orders'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('create-orders'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('update-order'))->handle($request, $next);
        })->only(['update']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('delete-order'))->handle($request, $next);
        })->only(['destroy']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('filter-orders'))->handle($request, $next);
        })->only(['getByReference']);
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {
            // dd($request->all());
            // saco el id del usuario principal
            $perPage = isset($request->per_page) ? $request->per_page : (isset($request->rowsPerPage) ? $request->rowsPerPage : 100);
            $from = $request->from ? $request->from . ' 00:00:00' : Carbon::now()->startOfYear();
            $to = $request->to ? $request->to . ' 23:59:59' : Carbon::now()->endOfYear();
            // select query
            $domicilios = Domicilio::select([
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.total as price',
                'domicilios.status as status',
                'domicilios.type as service',
                'domicilios.weight as weight',
                'domicilios.delivery as delivery',
                'domicilios.log_sender_id as sender_id',
                'domicilios.liquidated as liquidated',
                'domicilios.content as content',
                DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id) as city"),
                DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id) as zone"),
                DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
                'domicilios.created_at',
                'domicilios.updated_at',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT SUM(value) FROM domicilio_collections WHERE domicilio_collections.domicilio_id = domicilios.id) as collection"),
                DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id) as client"),
                DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
                DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),

                DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
                DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone")

            ])
            ->leftJoin('domiciliarios', 'domiciliarios.id', 'domicilios.domiciliario_id')
            ->whereBetween('domicilios.created_at', [$from, $to])
            ->orderBy('domicilios.id', 'DESC') // Ordena los registros por fecha de creación descendente.

            ->where('domicilios.log_sender_id', 'LIKE', '%' . $request->sender_id . '%');

            if (!is_null($request->status)) {
                $domicilios->where('domicilios.status', '=', $request->status);
            }

            if (!is_null($request->domiciliario_id)) {
                $domicilios->where('domicilios.domiciliario_id', 'LIKE', '%' . $request->domiciliario_id . '%');
            }

            if (!is_null($request->liquidated)) {
                $domicilios->where('domicilios.liquidated', 'LIKE', '%' . $request->liquidated . '%');
            }

            if (!is_null($request->city_id)) {
                $domicilios->where('domicilios.city_id', '=', $request->city_id);
            }

            if (!is_null($request->zone_id)) {
                $domicilios->where('domicilios.zone_id', '=', $request->zone_id);
            }

            $columns = [
                'service',
                'client',
                'domiciliario',
                'id',
                'reference',
                //'payment',
                'weight',
                'status',
                'sender',
                'client_city',
                'client_zone'
            ];
            // el search
            if (isset($request->search)) {
                $domicilios = $domicilios->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $request->search . "%'");
            }
            // si se loguea un domiciliario
            if ($request->user() and is_numeric($request->user()->domiciliario_id)) {
                $domicilios = $domicilios->where('domiciliario_id', $request->user()->domiciliario_id);
            }
            if ($request->user() and is_numeric($request->user()->log_sender_id)) {
                $domicilios = $domicilios->where('log_sender_id', $request->user()->log_sender_id);
            }

            $domicilios = $domicilios->paginate($perPage);

            return $this->successResponse($domicilios);

        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(DomicilioRequest $request, DomicilioServices $domicilioService)
    {
        DB::beginTransaction();
        try {
            $domicilio = $this->domicilioService->storeNewOrder($request->all());
            DB::commit();
            return $this->successResponse([
                'message' => 'Se ha creado el domicilio correctamente.',
                'data' => $domicilio,
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function show(Domicilio $domicilio)
    {
        //
        try {
            return Helper::response('success', new DomicilioResource($domicilio), 200);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function edit(Domicilio $domicilio)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function update(DomicilioRequest $request)
    {
        DB::beginTransaction();
        try {
            // get id of user main
            $mainId = $request->user()->getIdMain($request->user());
            // save domicilio
            $domicilio = $this->domicilioService->updateOrder($request->all());
            DB::commit();
            return Helper::response('success', $domicilio, 200);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * the Domiciliary.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function updateDomiciliary(Request $request)
    {
        DB::beginTransaction();
        try {
            // get id of user main
            $mainId = $request->user()->getIdMain($request->user());
            // save domicilio
            $domicilio = $this->domicilioService->updateDomiciliary($request->all());
            DB::commit();
            return Helper::response('success', $domicilio, 200);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Domicilio $domicilio, DomicilioServices $domicilioService)
    {
        //
        DB::beginTransaction();
        try {
            //save auditoria
            if (!$domicilio->status !== 'Pendiente' && $request->user()->type === 4) {
                return Helper::response('error', 'Solo puedes eliminar domicilios que esten pendientes.', 400);
            }
            $domicilio->Products()->delete();
            $domicilio->delete();
            DB::commit();
            return Helper::response('success', ['message' => 'Se ha eliminado el domicilio.', 'data' => $domicilio], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     *  change status in bulk
     */
    public function changeStatusInbulk(BulkStatus $request) {
        DB::beginTransaction();
        try {
            $status = $this->domicilioService->changeStatusOrder($request->items, $request->status);
            DB::commit();
            return $status;
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     *  change status in liquidado
     */
    public function changeStatusPaid(Request $request) {
        //dd($request->all());
        DB::beginTransaction();
        try {
            $status = $this->domicilioService->changeStatusPaid($request->id, $request->status_paid);
            DB::commit();
            return $status;
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    // Guia
    public function getByReference($id) {
        try {
            $data = Domicilio::with([
                'client',
                'domiciliario',
                'products',
                'logSender',
                'collections' => function($query) {
                    $query->orderBy('created_at', 'desc');
                },
                'statusDomicilio',
                'evidences',
                'products',
                'news'
            ])
            ->where('reference', $id)
            ->first();

            if (!$data) {
                return Helper::response('error', 'No se encontraron datos para la referencia especificada', 404);
            }
            return $this->successResponse($data);
        } catch(\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     *  upload evidence
     */
    public function uploadEvidence(UploadEvidence $request) {
        DB::beginTransaction();
        try {
            $evidence = $this->domicilioService->uploadEvidence($request->file, $request->domicilioId);
            DB::commit();
            return [
                'success' => true,
                'data' => $evidence
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * @gest last events
     */
    public function getLastOrders(Request $request) {
        try {
            $user = $request->user();
            $orders = $this->domicilioService->getLastorders($user);
            return [
                'success' => true,
                'data' => $orders
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Generate order guide
     */
    public function generateGuide(ReferenceRequest $request) {
        try {
            $guide = $this->domicilioService->generateGuide($request->reference);
            return [
                'success' => true,
                'data' => $guide
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * export excel for import
     *
     */
    public function exportExcel(Request $request) {
        try {
            return $this->domicilioService->exportExcel();
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * export excel for import
     *
     */
    public function importFromExcel(DomicilioImportRequest $request) {
        try {
            return [
                'success' => true,
                'data' => $this->domicilioService->importFromExcel($request->all()),
                'message' => 'Se ha importado el excel correctamente.'
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Create order
     */
    public function createOrder(OrderCreate $request) {
        try {
            return [
                'success' => true,
                'data' => $this->domicilioService->createOrder($request->all()),
                'message' => 'Se ha creado la orden correctamente.'
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Cancelar orden
     * @param $reference
     */
    public function cancelOrder($reference) {
        DB::beginTransaction();
        try {
            $order = $this->domicilioService->cancelOrder($reference);
            DB::commit();
            return [
                'success' => true,
                'data' => $order,
                'message' => 'Se ha cancelado la orden correctamente.'
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Filtra los estados y las novedades
     * @param $reference
     */
    public function showOrderDetails($reference) {
        try {
            return [
                'success' => true,
                'data' => $this->domicilioService->showOrderDetails($reference),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
