<?php
namespace App\Services\Domicilio;

use DB;
use PDF;
use App\Evidence;
use Carbon\Carbon;
use App\Domicilio;
use Dompdf\Options;
use App\Helpers\Helper;
use Illuminate\Support\Str;
use App\Traits\ApiResponser;
use App\Exports\DomicilioExcel;
use App\Imports\DomicilioImport;
use Maatwebsite\Excel\Facades\Excel;
use App\Http\Controllers\ClientController;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use App\Services\LogSender\LogSenderServices;
use App\Http\Resources\Domicilio\DomicilioResource;
use App\Http\Resources\Domicilio\DomicilioDataResource;

/**
 *
 */
class DomicilioServices
{
	use ApiResponser;

    protected $senderServices;
	function __construct() {
        $this->senderServices = new LogSenderServices();
	}

    //filter by ref domicilio
	public function filterByRef($reference, $mainId) {
		$issetDomicilio = Domicilio::where('reference', '=', $reference)->where('user_id', '=', $mainId)->first();
		if (isset($issetDomicilio)) {
			return new DomicilioResource($issetDomicilio);
		} else {
			return false;
		}
	}

    /**
     *  change status in bulk
     * @param array $items
     * @param string $status
     */
    public function changeStatusOrder(array $items, string $status) {
        try {
            $updateDomicilio = Domicilio::whereIn('id', $items)->update(['status' => $status]);
            $this->setStatuses($items, $status, '');
            return $this->successResponse([
                'success' => true,
                'data' => $updateDomicilio,
                'message' => 'Se ha cambiado el estado de los domicilios seleccionados.'
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     *  change status liqueidado
     * @param int $id
     * @param boolean $status_paid
     */
    public function changeStatusPaid($id, $status_paid) {
        try {
            $updateDomicilio = Domicilio::where('id', $id)->update(['paid' => $status_paid, 'liquidated' => $status_paid]);
            return $this->successResponse([
                'success' => true,
                'data' => $updateDomicilio,
                'message' => 'Se ha cambiado el estado de liquidado'
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * set statuses
     * @param array $items
     * @param string $status
     */
    public function setStatuses($items, $statusCode, $observation = '') {
        foreach ($items as $dom => $domicilio) {
            $domicilio = Domicilio::findOrFail($domicilio);
            if($statusCode == 'Entregado') {
                $domicilio->paid = 1;
                $domicilio->update();
            }
            $status = \App\Status::create([
                'status' => $statusCode,
                'observation' => $observation,
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);
        }
    }

    /**
     * Get status from order id
     * @param int $orderId
     */
    public function getStatusByOrderId(int $orderId) {
        try {
            $statuses = DB::table('statuses')
            ->select(
                'statuses.status as status',
                DB::raw("DATE_FORMAT(statuses.created_at, '%d/%m/%Y') as date"),
                DB::raw("DATE_FORMAT(statuses.created_at, '%h:%i:%s %p') as hour")
            )
            ->where('statuses.domicilio_id', $orderId)
            ->get();
            return $statuses;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * upload evidence
     * @param File $file
     * @param int $domicilioId
     * @param string $type
     */
    public function uploadEvidence($file, $domicilioId, $type = 'Entrega') {
        try {
            if (isset($file)) {
                $path = Helper::uploadImage($file, 'evidence');
                $file = url('images/evidence', $path);
                $evidence = Evidence::create([
                    'domicilio_id' => $domicilioId,
                    'path' => $file,
                    'type' => $type,
                ]);
                return $evidence;
            }
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Store new order
     * @param array $request
     */
    public function storeNewOrder($request) {
        try {
            // generate client or update
            $clientId = null;
            if (isset($request['client']) and isset($request['client']['id'])) {
                $clientId = $request['client']['id'];
                $clientId = ClientController::saveNewClient($request['client']);
            } else {
                $clientId = ClientController::saveNewClient($request['client']);
            }
            $request['zone_id'] = isset($request['client']['zone_id']) ? $request['client']['zone_id'] : null;
            $request['city_id'] = isset($request['client']['city_id']) ? $request['client']['city_id'] : null;
            $domicilio = Domicilio::create(array_merge($request, ['status' => 'Pendiente', 'client_id' => $clientId]));
            $reference = Str::padLeft($domicilio->id, 9, 0);
            $domicilio->update(['reference' => $reference]);
            // generate initial status
            $status = \App\Status::create([
                'status' => 'Pendiente',
                'observation' => isset($request['observation']) ? $request['observation'] : '',
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);
            // save products in orders
            foreach ($request['products'] as $prods => $product) {
                \App\DomicilioProduct::create([
                    'reference' => isset($product['reference']) ? $product['reference'] : '',
                    'name' => isset($product['name']) ? $product['name'] : '',
                    'price' => isset($product['price']) ? $product['price'] : '',
                    'quantity' => isset($product['quantity']) ? $product['quantity'] : '',
                    'domicilio_id' => $domicilio->id
                ]);
            }
            // return order
            return $domicilio;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * update domicilio
     * @param array $request
     */
    public function updateOrder($request) {
        try {
            // find domicilios
            $domicilio = Domicilio::findOrFail($request['id']);
            $request['zone_id'] = isset($request['client']['zone_id']) ? $request['client']['zone_id'] : null;
            $request['city_id'] = isset($request['client']['city_id']) ? $request['client']['city_id'] : null;
            // get cllient data
            $clientId = null;
            if (isset($request['client']) and isset($request['client']['id'])) {
                $clientId = $request['client']['id'];
                $clientId = ClientController::saveNewClient($request['client']);
            } else {
                $clientId = ClientController::saveNewClient($request['client']);
            }
            // get products
            $products = $request['products'];
            // fill nuew data
            unset($request['client']);
            unset($request['products']);
            isset($request['liquidated']) ? $domicilio->liquidated = 1 : $domicilio->liquidated = 0;
            isset($request['content']) ? $domicilio->content : '';
            $domicilio->update($request);
            // validamos los productos
            foreach ($products as $prods => $product) {
                if (isset($product['id'])) {
                    $productBd = \App\DomicilioProduct::findOrFail($product['id']);
                    $productBd->update($product);
                } else {
                    \App\DomicilioProduct::create([
                        'reference' => isset($product['reference']) ? $product['reference'] : '',
                        'name' => isset($product['name']) ? $product['name'] : '',
                        'price' => isset($product['price']) ? $product['price'] : '',
                        'quantity' => isset($product['quantity']) ? $product['quantity'] : '',
                        'domicilio_id' => $domicilio->id
                    ]);
                }
            }
            // return data
            return $domicilio;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * update domiciliary
     * @param array $request
     */
    public function updateDomiciliary($request) {
        try {
            // find domicilios
            $domicilio = Domicilio::findOrFail($request['id']);
            $domicilio->domiciliario_id = $request['domiciliary_id'];
            $domicilio->update();
            return $domicilio;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
    /**
     * Get last orders
     * @param $user
     */
    public function getLastorders($user) {
        try {
            $domicilios = Domicilio::select([
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.total as price',
                'domicilios.status as status',
                'domicilios.type as service',
                'domicilios.weight as weight',
                'domicilios.delivery as delivery',
                'domicilios.log_sender_id as sender_id',
                'domicilios.liquidated as liquidated',
                'domicilios.content as content',
                DB::raw("(SELECT name FROM cities WHERE id = domicilios.city_id) as city"),
                DB::raw("(SELECT name FROM zones WHERE id = domicilios.zone_id) as zone"),
                DB::raw('IF(domicilios.aditional_payment > 0, domicilios.aditional_payment, 0) as aditional_payment'),
                'domicilios.created_at',
                'domicilios.updated_at',
                DB::raw("(SELECT name FROM log_senders WHERE id = domicilios.log_sender_id) as sender"),
                DB::raw("(SELECT name FROM clients WHERE id = domicilios.client_id) as client"),
                DB::raw("(SELECT description FROM payment_methods WHERE id = domicilios.payment_method_id) as payment"),
                DB::raw("(SELECT name FROM domiciliarios WHERE id = domicilios.domiciliario_id) as domiciliario"),
                DB::raw("(SELECT name FROM cities WHERE id = (SELECT city_id FROM clients WHERE id = domicilios.client_id)) as client_city"),
                DB::raw("(SELECT name FROM zones WHERE id = (SELECT zone_id FROM clients WHERE id = domicilios.client_id)) as client_zone")
            ])
            ->leftJoin('domiciliarios', 'domiciliarios.id', 'domicilios.domiciliario_id')
            ->orderBy('domicilios.id', 'DESC')
            ->when($user['role_id'] === 4, function ($query) use($user) {
                return $query->where('log_sender_id', $user['log_sender_id']);
            })
            ->get()
            ->take(10);
            return $domicilios;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * generate order guide
     * @param string $orderReference
     */
    public function generateGuide(string $orderReference) {
        try {
            $domicilio = Domicilio::where('reference', $orderReference)->first();
			$fileName =  'guide_' . $domicilio->reference . '.' . 'pdf';
            if (ENV('APP_ENV') == 'production') {
                $url = url('guides');
            } else {
                $url = url('guides');
            }
            // si posee pdf lo devolvemos
            if ($domicilio->invoice_url) {
                return [
                    'file' => "{$url}/{$fileName}"
                ];
            }
            // prepare barcode and qr code
            $qrCode = QrCode::size(150)->generate($domicilio->reference);
            $configuration = \App\Configuration::all()->first();
            // generate pdf
            $pdf = PDF::loadView('onx/guides', array(
                'order' => $domicilio,
                'qrCode' => $qrCode,
                'itemsLabel' =>  $this->generateItemLabel($domicilio->Products),
                'configuration' => $configuration,
            ));
            $options = new Options();
            $options->set('isPhpEnabled', true);
            $options->set('isRemoteEnabled', true);
            $pdf->getDomPDF()->setPaper([0, 0, 280, 380])->setOptions($options);
			$path = public_path('guides/');
			$pdf->save($path . '/' . $fileName);
			$file = public_path() . "/guides/" . $fileName;
            $domicilio->update([
                'status' => 'Guía Generada',
                'invoice_url' => "{$url}/{$fileName}",

            ]);
            $status = \App\Status::create([
                'status' => 'Guía Generada',
                'observation' => '',
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);
            return [
                'file' => "{$url}/{$fileName}",
                'domicilio' => $domicilio
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Generate guide label products
     * @param array $products
     */
    public function generateItemLabel($products) {
        try {
            $label = '';
            // generate products label
            foreach ($products as $prod => $product) {
                $label .= "{$product->name} X{$product->quantity}";
                if ($prod >= 0 and $prod < (count($products) - 1)) {
                    $label .= ', ';
                }
                if ($prod == (count($products) - 1)) {
                    $label .= '.';
                }
            }

            // return label
            return $label;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * downlaod Excell
     */
    public function exportExcel() {
        try {
            return Excel::download(new DomicilioExcel, 'order_excel.xlsx');
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * impor from excel
     * @param array $params
     */
    public function importFromExcel(array $params) {
        try {
            $file = $params['file'];
            $filePath = $file->storeAs('temp', 'order_file.xlsx', 'public');
            $import = Excel::import(new DomicilioImport, storage_path('app/public/' . $filePath));
            \Storage::delete($filePath);
            return $import;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * get order items by id
     * @param int $id
     */
    public function getItemsById(int $id) {
        try {
            $items = DB::table('domicilio_products')
            ->select(DB::raw("CONCAT(domicilio_products.name, ' x ', domicilio_products.quantity) as item"))
            ->where('domicilio_products.domicilio_id', $id)
            ->get();
            return $items;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Create orders
     * @param array $params
     */
    public function createOrder(array $params) {
        DB::beginTransaction();
        try {
            // generate client data
            $clientId = ClientController::saveNewClient($params['client']);
            $client = \App\Client::find($clientId);
            // generate sender data
            $senderData = $this->senderServices->createSender($params['sender']);
            $params['log_sender_id'] = $senderData ? $senderData->id : null;
            // get order price by zone
            $zone = \App\Zone::findOrFail($client->zone_id);
            $params['delivery'] = !is_null($zone->finally_price) ? $zone->finally_price : ENV('ORDER_PRICE');
            // prepare zone and city id in domicilio
            $params['zone_id'] = $zone->id;
            $params['city_id'] = $client->city_id;
            $domicilio = Domicilio::create(array_merge($params, ['status' => 'Pendiente', 'client_id' => $client->id]));
            $domicilio->update(['reference' => Str::padLeft($domicilio->id, 9, 0)]);
            // generate initial status
            $status = \App\Status::create([
                'status' => 'Pendiente',
                'observation' => isset($params['observation']) ? $params['observation'] : '',
                'domicilio_id' => $domicilio->id,
                'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            ]);
            // save products in orders
            // foreach ($params['products'] as $prods => $product) {
            //     \App\DomicilioProduct::create([
            //         'reference' => isset($product['reference']) ? $product['reference'] : '',
            //         'name' => isset($product['name']) ? $product['name'] : '',
            //         'price' => isset($product['price']) ? $product['price'] : '',
            //         'quantity' => isset($product['quantity']) ? $product['quantity'] : '',
            //         'domicilio_id' => $domicilio->id
            //     ]);
            // }
            DB::commit();
            return $domicilio;
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Cancelamos la orden
     * @param $reference
     */
    public function cancelOrder($reference) {
        try {
            // obtenemos la orden
            $order = Domicilio::where('reference', $reference)->first();
            // validamos si existe
            if (!$order) {
                return $this->handlerException('No existe la orden que intentas cancelar.');
            }
            // validamos el estado actual de la orden
            if ($order->status != 'Pendiente' and $order->status != 'Guía Generada') {
                return $this->handlerException(
                    "La orden no puede ser cancelada ya que se encuentra en estado {$order->status}."
                );
            }
            // cancelamos la orden
            $order->update(['status' => 'Cancelado']);
            return $order;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Filtra los estados y las novedades
     * @param $reference
     */
    public function showOrderDetails($reference) {
        try {
            $order = Domicilio::where('reference', $reference)
            ->with(['statusDomicilio', 'News'])
            ->first();
            if (!$order) {
                return $this->handlerException('No existe la orden que intentas consultar.');
            }
            return new DomicilioDataResource($order);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
