<?php
namespace App\Services\LogSender;

use DB;
use App\User;
use App\LogSender;
use App\Traits\ApiResponser;
use App\Services\User\UserServices;

final class LogSenderServices
{
    use ApiResponser;

    public $userService;

    /**
     * construct
     */
    public function __construct() {
        $this->userService = new UserServices();
    }

    /**
     * List senders
     * @param int $page
     * @param int $perPage
     * @param string $search
     */
    public function listSenders(int $page = 1, int $perPage = 10, string $search, bool $filters = false ) {
        try {
            // columsn
            $columns = [
                'id',
                'name',
                'company',
                'address',
                'phone',
                'dni'
            ];
            // query
            $senders = DB::table('log_senders')
            ->select(
                'log_senders.name as name',
                'log_senders.company as company',
                'log_senders.address as address',
                'log_senders.phone as phone',
                'log_senders.dni as dni',
                'log_senders.tax_collection as tax_collection',
                'log_senders.order_price as order_price',
                'log_senders.discount as discount',
                'log_senders.fullfilment as fullfilment',
                'log_senders.id as id',
                'log_senders.email as email',
            )
            ->when(!is_null($search), function ($query) use ($search, $columns) {
                return $query->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
            })
            ->whereNull('log_senders.deleted_at');
            
            // return
            return $filters ? $senders->get() : $senders->paginate($perPage);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Create log senders
     * @param array $params
     */
    public function storeSender(array $params) {
        try {
            // create sender
            $data = LogSender::create($params);
            // create user
            if ($params['email'] and $params['password']) {
                $params = [
                    'name' => $params['email'],
                    'email' => $params['email'],
                    'password' => bcrypt($params['password']),
                    'type' => 4,
                    'is_active' => true,
                    'log_sender_id' => $data->id,
                    'role_id' => 4
                ];
                $this->userService->createUser($params);
            }
            // return data
            return $this->filterSenderById($data->id);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * fitler sender by id
     * @param int $id
     */
    public function filterSenderById(int $id) {
        try {
            // columsn
            $sender = DB::table('log_senders')
            ->select(
                'log_senders.name as name',
                'log_senders.company as company',
                'log_senders.address as address',
                'log_senders.phone as phone',
                'log_senders.dni as dni',
                'log_senders.id as id',
                'log_senders.email as email',
                'log_senders.tax_collection as tax_collection',
                'log_senders.order_price as order_price',
                'log_senders.discount as discount',
                'log_senders.fullfilment as fullfilment'
            )
            ->where('id', $id)
            ->whereNull('log_senders.deleted_at')
            ->first();
            // return
            return $sender;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Update sender
     * @param array $params
     */
    public function updateSender(array $params) {
        try {
            // update sender
            $data = LogSender::findOrFail($params['id']);
            $data->update($params);
            // create user
            if ($params['email']) {
                $userParam = [
                    'username' => $params['email'],
                    'email' =>  $params['email'],
                    'password' => isset($params['password']) ? $params['password'] : null,
                ];
                $this->userService->updateUserSender($data->id, $userParam);
            }
            // return data
            return $this->filterSenderById($data->id);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * delete sender
     * @param int $senderId
     */
    public function deleteSender(int $senderId) {
        try {
            // delete sender
            $data = LogSender::findOrFail($senderId);
            $data->delete();
            // borramos el usuario
            DB::statement('SET FOREIGN_KEY_CHECKS=0;');
            DB::table('users')->where('log_sender_id', '=', $data->id)->delete();
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            // return data
            return $senderId;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * get for select
     */
    public function getForSelect() {
        try {
            $data = DB::table('log_senders')
            ->select(
                'log_senders.id as value',
                'log_senders.name as label',
                'log_senders.order_price as order_price',
                'log_senders.discount as discount',
                'log_senders.fullfilment as fullfilment',
                'log_senders.tax_collection as tax_collection',
            )
            ->whereNull('deleted_at')
            ->get();
            return $data;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Create fron order
     * @param array $sender
     */
    public function createSender(array $params) {
        try {
            // filter by id
            $sender = \App\LogSender::where('dni', $params['dni'])->first();
            if ($sender) {
                $userParam = [
                    'username' => $params['email'],
                    'email' =>  $params['email'],
                    'password' => isset($params['dni']) ? $params['dni'] : null,
                ];
                $this->userService->updateUserSender($sender->id, $userParam);
                return $sender;
            }
            $params['order_price'] = ENV('OnexDelivery_ORDER_PRICE');
            $params['tax_collection'] = ENV('OnexDelivery_TAX_COLLECTION');
            // store data
            $sender = LogSender::create($params);
            // create user data from sender
            $params = [
                'name' => $params['email'],
                'email' => $params['email'],
                'password' => bcrypt($params['dni']),
                'type' => 4,
                'is_active' => true,
                'log_sender_id' => $sender->id,
                'role_id' => 4
            ];
            $this->userService->createUser($params);
            // return data
            return $sender;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
