<?php
namespace App\Services\Relations;

use DB;
use App\DispatchRelation;
use App\DomicilioCollection;
use App\Traits\ApiResponser;
use App\Services\Domicilio\DomicilioServices;

final class RelationDispathService
{
    use ApiResponser;

    public $domicilioService;

    /**
     * constructor
     */
    public function __construct() {
        $this->domicilioService = new DomicilioServices();
    }

    /**
     * List relations dispatch
     * @param string $start;
     * @param string $end;
     * @param int $page;
     * @param int $end;
     * @param int $perPage;
     * @param int $domiciliarioId
     */
    public function getRelationList(string $start, string $end, int $page, int $perPage, int $domiciliarioId) {
        try {
            $relations = DB::table('dispatch_relations')
            ->select(
                'dispatch_relations.id as id',
                'dispatch_relations.status as status',
                DB::raw('(DATE_FORMAT(dispatch_relations.created_at, "%d-%m-%Y")) as date'),
                DB::raw('IF((SELECT COUNT(dispatch_relation_items.id) from dispatch_relation_items WHERE dispatch_relation_items.dispatch_relation_id = dispatch_relations.id) > 0,
                (SELECT COUNT(dispatch_relation_items.id) from dispatch_relation_items WHERE dispatch_relation_items.dispatch_relation_id = dispatch_relations.id), 0) as totalItems')
            )
            ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
            ->whereBetween('dispatch_relations.created_at', [$start, $end])
            ->orderBy('dispatch_relations.created_at', 'DESC')
            ->paginate($perPage);
            return [
                'data' => $relations->items(),
                'lastPage' =>  $relations->lastPage()
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List relations dispatch
     * @param string $start;
     * @param string $end;
     * @param int $page;
     * @param int $end;
     * @param int $perPage;
     * @param int $relationId;
     * @param int $domiciliarioId
     */
    public function getOrderByRelationOrNot(string $start, string $end, int $page, int $perPage, $relationId, int $domiciliarioId, string $status) {
        try {
            $orders = DB::table('domicilios')
            ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
            ->leftJoin('cities', 'cities.id', 'clients.city_id')
            ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.payment_method_id')
            ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.domicilio_id', 'domicilios.id')
            ->leftJoin('dispatch_relations', 'dispatch_relations.id', 'dispatch_relation_items.dispatch_relation_id')
            ->select(
                "domicilios.total as collection",
                "domicilios.id as id",
                "domicilios.status as status",
                "clients.name as client",
                "clients.address as address",
                "clients.getho as getho",
                "cities.name as city",
                "domicilios.delivery as payment",
                DB::raw("CONCAT(LPAD(domicilios.id, 8, '0')) as reference"),
            )
            ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
            ->whereBetween('domicilios.created_at', [$start, $end])
            ->orWhereBetween('dispatch_relations.created_at', [$start, $end])
            ->when(!is_null($relationId), function ($query) use($relationId) {
                return $query->where('dispatch_relations.id', $relationId);
            })
            ->when($status !== 'todos', function ($query) use($status) {
                return $query->where('domicilios.status', $status);
            })
            ->groupBy('domicilios.id')
            ->paginate($perPage);
            return [
                'data' => $orders->items(),
                'lastPage' =>  $orders->lastPage()
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Filter order by id
     * @param int $id
     * @param Request $request
     */
    public function getOrderById(int $id) {
        try {
            $order = DB::table('domicilios')
            ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
            ->leftJoin('cities', 'cities.id', 'clients.city_id')
            ->leftJoin('log_senders', 'log_senders.id', 'domicilios.log_sender_id')
            ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.id')
            ->select(
                'domicilios.id as id',
                'domicilios.reference as reference',
                'domicilios.status as status',
                'domicilios.total as total',
                "clients.name as client",
                "clients.address as address",
                "clients.getho as getho",
                "cities.name as city",
                "clients.phone as phone",
                "log_senders.name as sender",
                "log_senders.phone as sender_phone",
                DB::raw("(IF(SUM(domicilio_collections.value) > 0, SUM(domicilio_collections.value), 0)) as total_collected")
            )
            ->where('domicilios.id', $id)
            ->first();
            $collections = DomicilioCollection::where('domicilio_id', $id)->get();
            if ($order) {
                return [
                    'order' => $order,
                    'collections' => $collections->sortByDesc('created_at')->values(),
                    'statuses' => $this->domicilioService->getStatusByOrderId($id),
                    'items' => $this->domicilioService->getItemsById($id),
                ];
            }
            return $this->handlerException('No existe una orden con este id.', 400);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List dispatch relation
     * @param int $perPage
     * @param string $from
     * @param string $to
     * @param string $search
     */
    public function listRelation(int $perPage, string $from, string $to, string $search)
	{
		//
		try {
            // $colums
            $columns = [
                'created_at',
                'status',
                'domiciliario_name',

            ];
            // do query return
			$dispatchRelations = DispatchRelation::select(
				'dispatch_relations.*',
				'domiciliarios.name as domiciliario_name',
			)
			->leftJoin('domiciliarios', 'domiciliarios.id', '=', 'dispatch_relations.domiciliario_id')
			->leftJoin('dispatch_relation_items', 'dispatch_relation_items.dispatch_relation_id', '=', 'dispatch_relations.id')
			->groupBy('dispatch_relations.id')
            ->whereBetween('dispatch_relations.created_at', [$from, $to])
            ->when(!is_null($search), function ($query) use($search, $columns) {
                $query->orHavingRaw("CONCAT_WS('', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
            })
            ->orderBy('dispatch_relations.created_at', 'DESC')
			->paginate($perPage);

			// Recorrer cada relación de despacho para calcular estadísticas - HERC
			foreach ($dispatchRelations as $dispatchRelation) {
				// Calcular el total de domicilios realizados. - HERC
				$dispatchRelation->totalItems = DB::table('dispatch_relation_items')
				->where('dispatch_relation_id', $dispatchRelation->id)
				->count();

				// Calcular la cantidad domicilios entregados. - HERC
				$dispatchRelation->delivered = DB::table('dispatch_relation_items')
				->join('domicilios', 'dispatch_relation_items.domicilio_id', '=', 'domicilios.id')
				->where('dispatch_relation_id', $dispatchRelation->id)
				->where('domicilios.status', 'Entregado')
				->count();

				// Obtener el valor del campo "total" de la tabla "domicilios" y se usa para mostrar el recaudo total. - HERC
				$dispatchRelation->recaudo = DB::table('domicilios')
				->join('dispatch_relation_items', 'domicilios.id', '=', 'dispatch_relation_items.domicilio_id')
				->where('dispatch_relation_items.dispatch_relation_id', $dispatchRelation->id)
				->sum('domicilios.total');
			}
			return $dispatchRelations;
		} catch (\Exception $e) {
			return $this->errorResponse($e->getMessage(), 400);
		}
	}

    /**
     * mark as complete
     * @param int $id
     */
    public function completeRelation(int $id) {
		try {
            $relation = DispatchRelation::findOrFail($id);
            if ($relation) {
                foreach ($relation->dispatchRelationItems as $key => $value) {
                    if (!is_null($value->domicilio) and $value->domicilio->status !== 'Entregado') {
                        return $this->handlerException("Aun queda pendiente por completar el domicilio #{$value->domicilio->reference}");
                    }
                }

                // change status
                $relation->update(['status' => 'Completada']);
                return $relation;
            }
            return $this->handlerException('No existe una relación con este id');
		} catch (\Exception $e) {
			return $this->handlerException($e->getMessage(), 400);
		}
    }
}
