<?php

namespace Tests\Feature;

use App\Causation;
use App\Provider;
use App\User;
use App\TypesCausationItem;
use Tests\TestCase;
use Illuminate\Support\Facades\Hash;

class CausationTest extends TestCase
{
    public function setUp(): void
    {
        parent::setUp();

        $user = User::where('email', 'causationTest@test.com')->first();

        if($user){
            $user->delete();
        }

        $user = new User;

        $data = array(
            'name' => 'causationTest',
            'email' => 'causationTest@test.com',
            'password' => Hash::make('test'),
            'type' => 1,
        );

        $user = $user::create($data);

        $this->actingAs($user, 'api');
    }

    public function test_can_list()
    {
        $this->json('GET', route('causations.index', ['search' => 'test']))
        ->assertStatus(200);
    }

    public function test_can_store()
    {
        $data = array(
            'provider_id' => Provider::first()->id,
            'date' => '2020-01-01',
            'items' => [
                [
                    'types_causation_item_id' => TypesCausationItem::first()->id,
                    'price' => 100
                ]
            ]
        );

        $causation = $this->json('POST', route('causations.store'), $data)
            ->assertStatus(201);

        $causation = json_decode($causation->content());

        Causation::find($causation->id)->delete();
    }

    public function test_cannot_store()
    {
        $data = array(
            'name' => ''
        );

        $this->json('POST', route('causations.store'), $data)
            ->assertStatus(422);
    }

    public function test_can_update()
    {
        $causation = new Causation;

        $data = array(
            'provider_id' => Provider::first()->id,
            'date' => '2020-01-01',
            'items' => [
                [
                    'types_causation_item_id' => TypesCausationItem::first()->id,
                    'price' => 100
                ]
            ]
        );

        $causation = $this->json('POST', route('causations.store'), $data);
        $causation = json_decode($causation->content());

        $data = array(
            'date' => '2020-01-02',
        );

        $this->json('PUT', route('causations.update', $causation->id), $data)
            ->assertStatus(200);

        Causation::find($causation->id)->delete();
    }

    public function test_cannot_update()
    {
        $causation = new Causation;

        $dataOld = array(
            'provider_id' => Provider::first()->id,
            'date' => '2020-01-01',
            'items' => [
                [
                    'types_causation_item_id' => TypesCausationItem::first()->id,
                    'price' => 100
                ]
            ]
        );

        $causation = $this->json('POST', route('causations.store'), $dataOld);
        $causation = json_decode($causation->content());

        $data = array(
            'date' => 'a'
        );

        $this->json('PUT', route('causations.update', $causation->id), $data)
            ->assertStatus(422);

        Causation::find($causation->id)->delete();
    }

    public function test_can_show()
    {
        $causation = new Causation;

        $data = array(
            'provider_id' => Provider::first()->id,
            'date' => '2020-01-01',
            'items' => [
                [
                    'types_causation_item_id' => TypesCausationItem::first()->id,
                    'price' => 100
                ]
            ]
        );

        $causation = $this->json('POST', route('causations.store'), $data);
        $causation = json_decode($causation->content());

        $this->json('GET', route('causations.show', $causation->id))
        ->assertStatus(200);

        Causation::find($causation->id)->delete();
    }

    public function test_can_delete()
    {
        $causation = new Causation;

        $data = array(
            'provider_id' => Provider::first()->id,
            'date' => '2020-01-01',
            'items' => [
                [
                    'types_causation_item_id' => TypesCausationItem::first()->id,
                    'price' => 100
                ]
            ]
        );

        $causation = $this->json('POST', route('causations.store'), $data);
        $causation = json_decode($causation->content());

        $this->json('DELETE', route('causations.destroy', $causation->id))
        ->assertStatus(200);
    }

    public function test_cannot_delete()
    {
        $this->json('DELETE', route('causations.destroy', 0))
        ->assertStatus(404);
    }
}
