<?php

namespace App\Http\Controllers;

use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

use App\Domicilio;

use App\Http\Requests\Domicilio\BulkStatus;
use Illuminate\Http\Request;
use App\Helpers\Helper;
use App\Http\Requests\Domicilio\DomicilioRequest;
use App\Http\Resources\Domicilio\DomicilioResource;
use App\Services\Domicilio\DomicilioServices;
use App\Traits\ApiResponser;
use App\Services\User\UserServices;
use App\Http\Requests\Domicilio\OrderCreate;
use App\Http\Requests\Domicilio\UploadEvidence;
use App\Http\Requests\Domicilio\ReferenceRequest;
use App\Http\Middleware\PermissionsMiddleware;
use App\Http\Requests\Domicilio\DomicilioImportRequest;
use App\Http\Requests\Domicilio\FindOrderRequest;
use App\Http\Requests\Domicilio\FindOrdersDomiciliarioRequest;
use App\Http\Requests\Domicilio\GuideRequest;
use setasign\Fpdi\Fpdi;
use App\Http\Requests\Domicilio\LastOrdersRequest;

class DomicilioController extends Controller
{
    use ApiResponser;
    protected $userService, $domicilioService;

    public function __construct(UserServices $userService, DomicilioServices $domicilioService)
    {
        $this->userService = $userService;
        $this->domicilioService = $domicilioService;
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('last-order'))->handle($request, $next);
        })->only(['getLastOrders']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('list-orders'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('create-orders'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('update-order'))->handle($request, $next);
        })->only(['update']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('delete-order'))->handle($request, $next);
        })->only(['destroy']);
        /*$this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('filter-orders'))->handle($request, $next);
        })->only(['getByReference']); */
    }


    public function index(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 100);
            $domicilios = $this->domicilioService->getDomiciliosForUser($request);

            return $this->successResponse($domicilios);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(DomicilioRequest $request, DomicilioServices $domicilioService)
    {
        DB::beginTransaction();
        try {
            $domicilio = $this->domicilioService->storeNewOrder($request->all());
            DB::commit();
            return $this->successResponse([
                'message' => 'Se ha creado el domicilio correctamente.',
                'data' => $domicilio,
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function show(Domicilio $domicilio)
    {
        try {
            return Helper::response('success', new DomicilioResource($domicilio), 200);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function edit(Domicilio $domicilio)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function update(DomicilioRequest $request)
    {
        DB::beginTransaction();
        try {
            $mainId = $request->user()->getIdMain($request->user());
            $domicilio = $this->domicilioService->updateOrder($request->all());
            $domicilio->update(['invoice_url' => null]);
            DB::commit();
            return Helper::response('success', $domicilio, 200);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * the Domiciliary.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function updateDomiciliary(Request $request)
    {
        DB::beginTransaction();
        try {
            // get id of user main
            $mainId = $request->user()->getIdMain($request->user());
            // save domicilio
            $domicilio = $this->domicilioService->updateDomiciliary($request->all());


            DB::commit();
            return Helper::response('success', $domicilio, 200);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Domicilio  $domicilio
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Domicilio $domicilio, DomicilioServices $domicilioService)
    {
        //
        DB::beginTransaction();
        try {
            //save auditoria
            if (!$domicilio->status !== 'Pendiente' && $request->user()->type === 4) {
                return Helper::response('error', 'Solo puedes eliminar domicilios que esten pendientes.', 400);
            }
            $domicilio->Products()->delete();
            $domicilio->delete();
            DB::commit();
            return Helper::response('success', ['message' => 'Se ha eliminado el domicilio.', 'data' => $domicilio], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     *  change status in bulk
     */
    public function changeStatusInbulk(BulkStatus $request)
    {
        DB::beginTransaction();
        try {
            $observation = $request->observation ?? '';
            $status = $this->domicilioService->changeStatusOrder(
                $request->items,
                $request->status,
                $observation,
                $request->date,
                $request->startTime,
                $request->endTime
            );
            DB::commit();
            return $status;
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     *  change status in liquidado
     */
    public function changeStatusPaid(Request $request)
    {
        //dd($request->all());
        DB::beginTransaction();
        try {
            $status = $this->domicilioService->changeStatusPaid($request->id, $request->status_paid);
            DB::commit();
            return $status;
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    // Guia
    public function getByReference($id)
    {
        try {
            $data = Domicilio::with([
                'client',
                'domiciliario',
                'products',
                'logSender',
                'collections' => function ($query) {
                    $query->orderBy('created_at', 'desc');
                },
                'statusDomicilio',
                'evidences',
                'products',
                'news' => function ($query) {
                    $query->with("type");
                },
                'dropshipper'
            ])
                ->where('reference', $id)
                ->first();

            if (!$data) {
                return Helper::response('error', 'No se encontraron datos para la referencia especificada', 404);
            }
            return $this->successResponse($data);
        } catch (\Exception $e) {
            return Helper::response('error', $e->getMessage(), 400);
        }
    }

    /**
     *  upload evidence
     */
    public function uploadEvidence(UploadEvidence $request)
    {
        DB::beginTransaction();
        try {
            $evidence = $this->domicilioService->uploadEvidence($request->file, $request->domicilioId);
            DB::commit();
            return [
                'success' => true,
                'data' => $evidence
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * @gest last events
     */
    public function getLastOrders(LastOrdersRequest $request)
    {
        try {
            $user = $request->user();
            $status = $request->status;
            $orders = $this->domicilioService->getLastorders($user, $status);
            return [
                'success' => true,
                'data' => $orders
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function getLastOrdersAll(Request $request)
    {
        try {
            $user = $request->user();
            $orders = $this->domicilioService->getLastordersAll($user);
            return [
                'success' => true,
                'data' => $orders
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function getLastOrdersApk(Request $request)
    {
        try {
            $user = $request->user();
            $status = $request->input('status', '');
            $unassigned = $request->input('unassigned', '0');
            $domiciliarios =  Auth::user()->domiciliario_id;

            if ($unassigned == '1') {
                $domiciliarios =  null;
            }

            $orders = $this->domicilioService->getLastordersNew($domiciliarios, $status);
            return [
                'success' => true,
                'data' => $orders
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }





    /**
     * Generate order guides
     */
    public function generateGuide(ReferenceRequest $request)
    {
        try {
            $guide = $this->domicilioService->generateGuide($request->reference);
            return [
                'success' => true,
                'data' => $guide
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function generateGuideCheck(GuideRequest $request)
    {
        try {
            $references = $request->input('orderReferences');
            $guides = [];

            // Llama al servicio para generar las guías (asegúrate de que esto devuelva las rutas correctas)
            foreach ($references as $reference) {
                $guide = $this->domicilioService->generateGuideCheck([$reference]);
                $guides[] = $guide['data'][0]['file']; // Asegúrate de que esto se ajuste a la estructura de datos correcta
            }

            // Combina los archivos PDF en uno solo
            $mergedPdf = $this->mergePdfs($guides);

            return [
                'success' => true,
                'file' => $mergedPdf
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    private function mergePdfs(array $pdfFiles)
    {
        $pdf = new Fpdi();

        // Establecer las dimensiones de la página horizontal (A4)
        $width = 297;  // Ancho A4 en orientación horizontal
        $height = 210; // Alto A4 en orientación horizontal

        // Agregar una nueva página para cada par de PDFs
        for ($i = 0; $i < count($pdfFiles); $i += 2) {
            $pdf->AddPage('L', [$width, $height]); // Agregar nueva página horizontal

            // Manejar el primer archivo (izquierda)
            $filePath1 = public_path('guides/' . basename($pdfFiles[$i]));
            if (file_exists($filePath1)) {
                $pageCount1 = $pdf->setSourceFile($filePath1);
                if ($pageCount1 > 0) {
                    $templateId1 = $pdf->importPage(1); // Asumiendo que solo necesitas la primera página
                    $pdf->useTemplate($templateId1, 0, 0, $width / 2, $height); // Izquierda
                }
            }

            // Manejar el segundo archivo (derecha), si existe
            if ($i + 1 < count($pdfFiles)) {
                $filePath2 = public_path('guides/' . basename($pdfFiles[$i + 1]));
                if (file_exists($filePath2)) {
                    $pageCount2 = $pdf->setSourceFile($filePath2);
                    if ($pageCount2 > 0) {
                        $templateId2 = $pdf->importPage(1); // Asumiendo que solo necesitas la primera página
                        $pdf->useTemplate($templateId2, $width / 2, 0, $width / 2, $height); // Derecha
                    }
                }
            }
        }

        // Guardar el PDF combinado en una ubicación
        $mergedFileName = 'combined_guide_' . time() . '.pdf';
        $mergedFilePath = public_path('guides/' . $mergedFileName);
        $pdf->Output($mergedFilePath, 'F');

        // Devolver la URL del archivo combinado
        return url('api/public/guides/' . $mergedFileName);
    }


    /**
     * export excel for import
     *
     */
    public function exportExcel(Request $request)
    {
        try {
            return $this->domicilioService->exportExcel();
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * export excel for import
     *
     */
    public function importFromExcel(DomicilioImportRequest $request)
    {
        try {
            return [
                'success' => true,
                'data' => $this->domicilioService->importFromExcel($request->all()),
                'message' => 'Se ha importado el excel correctamente.'
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Create order
     */
    public function createOrder(OrderCreate $request)
    {
        try {
            return [
                'success' => true,
                'data' => $this->domicilioService->createOrder($request->all()),
                'message' => 'Se ha creado la orden correctamente.'
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Cancelar orden
     * @param $reference
     */
    public function cancelOrder($reference)
    {
        DB::beginTransaction();
        try {
            $order = $this->domicilioService->cancelOrder($reference);
            DB::commit();
            return [
                'success' => true,
                'data' => $order,
                'message' => 'Se ha cancelado la orden correctamente.'
            ];
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Filtra los estados y las novedades
     * @param $reference
     */
    public function showOrderDetails($reference)
    {
        try {
            return [
                'success' => true,
                'data' => $this->domicilioService->showOrderDetails($reference),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function createOrderTempEcomdex(Request $request)
    {
        try {
            $data = $request->all();
            return [
                'success' => true,
                'data' => $this->domicilioService->createOrderTemp($data),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function findOrder(FindOrderRequest $request)
    {
        try {
            $data = $request->all();
            $search = $data["search"];
            return [
                'success' => true,
                'data' => $this->domicilioService->findOrder($search),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }


    public function findOrdersByDomiciliario(FindOrdersDomiciliarioRequest $request)
    {
        try {
            $domiciliarioId = $request->route('domiciliario_id');
            $from = $request->from ? $request->from . ' 00:00:00' : null;
            $to = $request->to ? $request->to . ' 23:59:59' : null;
            $perPage = $request->per_page ? $request->per_page : 10;
            return $this->domicilioService->findOrdersByDomiciliario($domiciliarioId, $perPage, $from, $to);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
