<?php

namespace App\Http\Controllers;


use App\Http\Requests\Settlements\StoreSettlementDomicilios;
use App\Http\Requests\Settlements\GenerateSettlementPDFRequest;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Services\Settlement\SettlementService;
use App\Traits\ApiResponser;

class SettlementDomiciliosController extends Controller
{
    protected $settlementService;
    use ApiResponser;

    public function __construct(SettlementService $settlementService)
    {
        $this->settlementService = $settlementService;
    }

    public function index(Request $request)
    {
        try {

            $perPage = $request->perPage ? $request->perPage : 10;
            $from = $request->from ? $request->from : Carbon::now()->startOfMonth();
            $to = $request->to ? $request->to . ' 23:59:59' : Carbon::now()->endOfMonth();
            $search = $request->search ? $request->search : '';
            $senderId = $request->sender_id;

            $settlements = $this->settlementService->listDomiciliosSettlements(
                $perPage,
                $from,
                $to,
                $search,
                $senderId
            );

            return $settlements;
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 500);
        }
    }

    public function show($id)
    {
        try {

            $pdfUrl = $this->settlementService->getSettlementInformation($id);

            return $this->successCreatedResponse([
                "message" => "Get domilicios settlements success",
                "data" => $pdfUrl
            ]);
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 500);
        }
    }

    public function viewPDF(GenerateSettlementPDFRequest $request)
    {
        try {

            DB::beginTransaction();

            $body = $request->all();
            $pdfUrl = $this->settlementService->getPDFByDomicilio(
                $body["settlement_id"],
                $body["domicilios"],
                $body["sender_id"] ?? null
            );

            DB::commit();

            return $this->successCreatedResponse([
                "message" => "Get domilicios settlements success",
                "data" => $pdfUrl
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse($e->getMessage(), 500);
        }
    }

    public function makeSettlementDomicilios(StoreSettlementDomicilios $request)
    {
        DB::beginTransaction();
        try {
            $settlement = $this->settlementService->changeDomiciliosToSettlement($request->items);
            DB::commit();

            return $this->successResponse([
                'message' => 'Se han liquidado los domicilios satisfactoriamente',
                'data' => $settlement
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    public function deleteDomicilioInSettlement($id)
    {
        DB::beginTransaction();
        try {
            $domicilio = $this->settlementService->removeDomicilioInSettlement($id);
            DB::commit();

            return $this->successResponse([
                "message" => "Se ha eliminado el domicilio {$domicilio->reference} de la liquidacion",
                "data" => $domicilio
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse($e->getMessage(), 500);
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $settlement = $this->settlementService->removeSettlementDomicilio($id);
            DB::commit();

            return $this->successResponse([
                "message" => "Se ha removido la liquidacion {$settlement->reference}",
                "data" => $settlement
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse($e->getMessage(), 500);
        }
    }
}
