<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use App\Zone;
use App\Traits\ApiResponser;
use Illuminate\Http\Request;
use App\Services\Zone\ZoneServices;
use App\Http\Requests\Zone\ZoneRequest;
use App\Http\Requests\Zone\UpdateZoneRequest;
use App\Http\Requests\Zone\ListByCityRequest;
use App\Http\Middleware\PermissionsMiddleware;

class ZoneController extends Controller
{
    use ApiResponser;

    public $service;

    /**
     * construct
     */
    public function __construct()
    {
        $this->service = new ZoneServices();
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('zone-list'))->handle($request, $next);
        })->only(['index']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('zone-delete'))->handle($request, $next);
        })->only(['destroy']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('zone-create'))->handle($request, $next);
        })->only(['store']);
        $this->middleware(function ($request, $next) {
            return (new PermissionsMiddleware('zone-update'))->handle($request, $next);
        })->only(['update']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //
        try {
            $page = $request->page ? $request->page : 1;
            $perPage = $request->perPage ? $request->perPage : 10;
            $search = $request->search ? $request->search : '';
            return $this->successResponse([
                'success' => true,
                'data' => $this->service->listZones($request->cityId, $page, $perPage, $search),
                'message' => 'Se ha creado la zona correctamente.'
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(ZoneRequest $request)
    {
        //
        DB::beginTransaction();
        try {
            $data = $this->service->storeZone($request->all());
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'data' => $data,
                'message' => 'Se ha creado la zona correctamente.'
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Zone  $zone
     * @return \Illuminate\Http\Response
     */
    public function show(Zone $zone)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Zone  $zone
     * @return \Illuminate\Http\Response
     */
    public function edit(Zone $zone)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Zone  $zone
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateZoneRequest $request, Zone $zone)
    {
        //
        DB::beginTransaction();
        try {
            $data = $this->service->updateZone($zone, $request->all());
            DB::commit();
            return $data;
        } catch (\Throwable $th) {
            DB::rollback();
            return $this->handlerException($th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Zone  $zone
     * @return \Illuminate\Http\Response
     */
    public function destroy(Zone $zone)
    {
        //
        DB::beginTransaction();
        try {
            $data = $this->service->deleteZone($zone);
            DB::commit();
            return $this->successResponse([
                'success' => true,
                'data' => $data,
                'message' => 'Se ha eliminado la zona correctamente.'
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * export cities to excell
     */
    public function exportExcel()
    {
        try {
            return $this->service->exportExcel();
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List by cities
     * @param Request $request
     */
    public function listZonesByCity(ListByCityRequest $request)
    {
        try {
            $page = $request->page ? $request->page : 1;
            $perPage = $request->perPage ? $request->perPage : 10;
            $sender = $request->sender;

            if(isset($sender)) {
                return $this->service->listZonesByCityWithLogSender(
                    $request->city,
                    $sender,
                    $page,
                    $perPage
                );
            }else {
                return $this->service->listZonesByCity(
                    $request->city,
                    $page,
                    $perPage
                );
            }
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
