<?php
namespace App\Services\City;

use DB;
use App\City;
use App\Traits\ApiResponser;
use App\Exports\CitiesExport;
use Maatwebsite\Excel\Facades\Excel;

class CityServices
{
    use ApiResponser;

    /**
     * store city
     * @param array $params
     */
    public function saveCity(array $params) {
        try {
            $city = City::create($params);
            return $this->filterCityById($city->id);
        } catch (\Exception $e) {
            return $this->handlerEception($e->getMessage());
        }
    }

    /**
     * update city
     * @param array $params
     */
    public function updateCity(City $city, array $params) {
        try {
            $city->update($params);
            return $this->filterCityById($city->id);
        } catch (\Exception $e) {
            return $this->handlerEception($e->getMessage());
        }
    }

    /**
     * List cities
     * @param int $page
     * @param int $perPage
     * @param string $search
     */
    public function getCity(int $page, int $perPage, string $search) {
        try {
            // columnas
            $columns = [
                'id',
                'name',
                'abbreviation',
                'departament',
                'city_deparment',
                'country'
            ];
            // do query
            $cities = DB::table('cities')
            ->leftJoin('departaments', 'departaments.id', 'cities.departament_id')
            ->select(
                'cities.id as id',
                'cities.name as name',
                'cities.country as country',
                'cities.abbreviation as abbreviation',
                'cities.departament_id as departament_id',
                'departaments.name as departament',
                DB::raw("CONCAT(cities.name, ', ', departaments.name) as city_deparment"),
            )
            ->when(!is_null($search), function ($query) use($search, $columns) {
                return $query
                ->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
            })
            ->paginate($perPage);
            // return
            return [
                'items' => $cities->items(),
                'totalItems' => $cities->total(),
                'page' => $page,
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * filter city by id
     * @param int $id
     */
    public function filterCityById(int $id) {
        try {
            $city = DB::table('cities')
            ->leftJoin('departaments', 'departaments.id', 'cities.departament_id')
            ->select(
                'cities.id as id',
                'cities.name as name',
                'cities.country as country',
                'cities.abbreviation as abbreviation',
                'cities.departament_id as departament_id',
                'departaments.name as departament',
                DB::raw("CONCAT(cities.name, ', ', departaments.name) as city_deparment"),
            )
            ->where('cities.id', $id)
            ->first();
            return $city;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * delete city
     * @param int $id
     */
    public function deleteCity(int $city) {
        try {
            $city = City::findOrFail($city)->delete();
            return $city;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * downlaod Excell
     */
    public function exportExcel() {
        try {
            return Excel::download(new CitiesExport, 'cities.xlsx');
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
