<?php

namespace App\Services\Relations;


use App\DispatchRelation;
use App\Domicilio;
use App\DomicilioCollection;
use App\Enums\DomicilioStatusConst;
use App\Traits\ApiResponser;
use App\Services\Domicilio\DomicilioServices;
use Illuminate\Support\Facades\DB;
use App\Enums\TabStatusConst;
use PDF;
use App\Configuration;

final class RelationDispathService
{
    use ApiResponser;

    public $domicilioService;

    /**
     * constructor
     */
    public function __construct(DomicilioServices $domicilioService)
    {
        $this->domicilioService = $domicilioService;
    }

    /**
     * List relations dispatch
     * @param string $start;
     * @param string $end;
     * @param int $page;
     * @param int $end;
     * @param int $perPage;
     * @param int $domiciliarioId
     */
    public function getRelationList(string $start, string $end, int $page, int $perPage, int $domiciliarioId)
    {
        try {
            $relations = DB::table('dispatch_relations')
                ->select(
                    'dispatch_relations.id as id',
                    'dispatch_relations.status as status',
                    'dispatch_relations.is_open as is_open',
                    DB::raw('(DATE_FORMAT(dispatch_relations.created_at, "%d-%m-%Y")) as date'),
                    DB::raw('IF((SELECT COUNT(dispatch_relation_items.id) from dispatch_relation_items WHERE dispatch_relation_items.dispatch_relation_id = dispatch_relations.id) > 0,
                (SELECT COUNT(dispatch_relation_items.id) from dispatch_relation_items WHERE dispatch_relation_items.dispatch_relation_id = dispatch_relations.id), 0) as totalItems')
                )
                ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
                ->whereBetween('dispatch_relations.created_at', [$start, $end])
                ->orderBy('dispatch_relations.created_at', 'DESC')
                ->paginate($perPage);
            return [
                'data' => $relations->items(),
                'lastPage' =>  $relations->lastPage()
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List relations dispatch
     * @param string $start;
     * @param string $end;
     * @param int $page;
     * @param int $end;
     * @param int $perPage;
     * @param int $relationId;
     * @param int $domiciliarioId
     */
    public function getOrderByRelationOrNot(string $start, string $end, int $page, int $perPage, $relationId, int $domiciliarioId, string $status)
    {
        try {
            $orders = DB::table('domicilios')
                ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
                ->leftJoin('cities', 'cities.id', 'clients.city_id')
                ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.payment_method_id')
                ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.domicilio_id', 'domicilios.id')
                ->leftJoin('dispatch_relations', 'dispatch_relations.id', 'dispatch_relation_items.dispatch_relation_id')
                ->select(
                    "domicilios.total as collection",
                    "domicilios.id as id",
                    "domicilios.status as status",
                    "clients.name as client",
                    "clients.address as address",
                    "clients.getho as getho",
                    "cities.name as city",
                    "domicilios.delivery as payment",
                    DB::raw("CONCAT(LPAD(domicilios.id, 8, '0')) as reference"),
                )
                ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
                ->whereBetween('domicilios.created_at', [$start, $end])
                ->orWhereBetween('dispatch_relations.created_at', [$start, $end])
                ->when(!is_null($relationId), function ($query) use ($relationId) {
                    return $query->where('dispatch_relations.id', $relationId);
                })
                ->when($status !== 'todos', function ($query) use ($status) {
                    return $query->where('domicilios.status', $status);
                })
                ->groupBy('domicilios.id')
                ->paginate($perPage);
            return [
                'data' => $orders->items(),
                'lastPage' =>  $orders->lastPage()
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    public function getOrdersByRelationAndStatus(
        string $start,
        string $end,
        int $page,
        int $perPage,
        $relationId,
        int $domiciliarioId,
        string $status
    ) {
        try {
            if ($status == TabStatusConst::PENDING) {
                $orders = $this->getDomiciliosByTabStatusPending($perPage, $domiciliarioId, $relationId);
                return [
                    'data' => $orders->items(),
                    'lastPage' =>  $orders->lastPage()
                ];
            } else if ($status == TabStatusConst::COMPLETED) {
                $orders = $this->getDomiciliosCompleted($perPage, $domiciliarioId, $relationId);
                return [
                    'data' => $orders->items(),
                    'lastPage' =>  $orders->lastPage()
                ];
            } else if ($status == TabStatusConst::WITHOUT_ASSIGN) {
                $orders = $this->getDomiciliosByTabStatusWithoutAssing($perPage, $domiciliarioId, $relationId);
                return [
                    'data' => $orders->items(),
                    'lastPage' =>  $orders->lastPage()
                ];
            } else {
                $orders = $this->getAllDomiciliosByRelation($perPage, $domiciliarioId, $relationId);
                return [
                    'data' => $orders->items(),
                    'lastPage' =>  $orders->lastPage()
                ];
            }
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 500);
        }
    }

    /**
     * Filter order by id
     * @param int $id
     * @param Request $request
     */
    public function getOrderById(string $id)
    {
        try {
            $order = DB::table('domicilios')
                ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
                ->leftJoin('cities', 'cities.id', 'clients.city_id')
                ->leftJoin('log_senders', 'log_senders.id', 'domicilios.log_sender_id')
                ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.id')
                ->select(
                    'domicilios.id as id',
                    'domicilios.reference as reference',
                    'domicilios.status as status',
                    'domicilios.total as total',
                    'domicilios.content as content',
                    'domicilios.location as location',
                    'domicilios.observation as observation',
                    'domicilios.domiciliario_id as domiciliario_id',
                    "clients.name as client",
                    "clients.address as address",
                    "clients.getho as getho",
                    "cities.name as city",
                    "clients.phone as phone",
                    "log_senders.name as sender",
                    "log_senders.phone as sender_phone",
                    DB::raw("(IF(SUM(domicilio_collections.value) > 0, SUM(domicilio_collections.value), 0)) as total_collected")
                )
                ->where('domicilios.id', $id)
                ->first();
            $collections = DomicilioCollection::where('domicilio_id', $order->id)->get();
            if ($order) {
                return [
                    'order' => $order,
                    'collections' => $collections->sortByDesc('created_at')->values(),
                    'statuses' => $this->domicilioService->getStatusByOrderId($order->id),
                    'items' => $this->domicilioService->getItemsById($order->id),
                ];
            }
            return $this->handlerException('No existe una orden con este id.', 400);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * Filter order by reference
     * @param int $reference
     * @param Request $request
     */
    public function getOrderByReference(string $reference)
    {
        try {
            $order = DB::table('domicilios')
                ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
                ->leftJoin('cities', 'cities.id', 'clients.city_id')
                ->leftJoin('log_senders', 'log_senders.id', 'domicilios.log_sender_id')
                ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.id')
                ->select(
                    'domicilios.id as id',
                    'domicilios.reference as reference',
                    'domicilios.status as status',
                    'domicilios.total as total',
                    'domicilios.content as content',
                    'domicilios.location as location',
                    'domicilios.observation as observation',
                    "clients.name as client",
                    "clients.address as address",
                    "clients.getho as getho",
                    "cities.name as city",
                    "clients.phone as phone",
                    "log_senders.name as sender",
                    "log_senders.phone as sender_phone",
                    DB::raw("(IF(SUM(domicilio_collections.value) > 0, SUM(domicilio_collections.value), 0)) as total_collected")
                )
                ->where('domicilios.reference', $reference)
                ->first();
            $collections = DomicilioCollection::where('domicilio_id', $order->id)->get();
            if ($order) {
                return [
                    'order' => $order,
                    'collections' => $collections->sortByDesc('created_at')->values(),
                    'statuses' => $this->domicilioService->getStatusByOrderId($order->id),
                    'items' => $this->domicilioService->getItemsById($order->id),
                ];
            }
            return $this->handlerException('No existe una orden con este id.', 400);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List dispatch relation
     * @param int $perPage
     * @param string $from
     * @param string $to
     * @param string $search
     */
    public function listRelation(int $perPage, string $from, string $to, string $search)
    {
        //
        try {
            // $colums
            $columns = [
                'created_at',
                'status',
                'domiciliario_name',

            ];
            // do query return
            $dispatchRelations = DispatchRelation::select(
                'dispatch_relations.*',
                'domiciliarios.name as domiciliario_name'
            )
                ->leftJoin('domiciliarios', 'domiciliarios.id', '=', 'dispatch_relations.domiciliario_id')
                ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.dispatch_relation_id', '=', 'dispatch_relations.id')
                ->groupBy('dispatch_relations.id')
                ->whereBetween('dispatch_relations.created_at', [$from, $to])
                ->when(!is_null($search), function ($query) use ($search, $columns) {
                    $query->orHavingRaw("CONCAT_WS('', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
                })
                ->orderBy('dispatch_relations.created_at', 'DESC')
                ->paginate($perPage);

            // Recorrer cada relaci贸n de despacho para calcular estad铆sticas - HERC
            foreach ($dispatchRelations as $dispatchRelation) {
                // Calcular el total de domicilios realizados. - HERC
                $dispatchRelation->totalItems = DB::table('dispatch_relation_items')
                    ->where('dispatch_relation_id', $dispatchRelation->id)
                    ->count();

                // Calcular la cantidad domicilios entregados. - HERC
                $dispatchRelation->delivered = DB::table('dispatch_relation_items')
                    ->join('domicilios', 'dispatch_relation_items.domicilio_id', '=', 'domicilios.id')
                    ->where('dispatch_relation_id', $dispatchRelation->id)
                    ->where('domicilios.status', 'Entregado')
                    ->count();

                // Obtener el valor del campo "total" de la tabla "domicilios" y se usa para mostrar el recaudo total. - HERC
                $dispatchRelation->recaudo = DB::table('domicilios')
                    ->join('dispatch_relation_items', 'domicilios.id', '=', 'dispatch_relation_items.domicilio_id')
                    ->where('dispatch_relation_items.dispatch_relation_id', $dispatchRelation->id)
                    ->sum('domicilios.total');
            }
            return $dispatchRelations;
        } catch (\Exception $e) {
            return $this->errorResponse($e->getMessage(), 400);
        }
    }

    /**
     * mark as complete
     * @param int $id
     */
    public function completeRelation(int $id)
    {
        try {
            $relation = DispatchRelation::findOrFail($id);
            if ($relation) {
                foreach ($relation->dispatchRelationItems as $key => $value) {
                    if (!is_null($value->domicilio) and $value->domicilio->status !== 'Entregado') {
                        return $this->handlerException("Aun queda pendiente por completar el domicilio #{$value->domicilio->reference}");
                    }
                }

                // change status
                $relation->update(['status' => 'Completada']);
                return $relation;
            }
            return $this->handlerException('No existe una relaci贸n con este id');
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    private function getDomiciliosByTabStatusPending($perPage, $domiciliarioId, $relationId)
    {
        /*
         * SE DEBE TRAER DOMICILIOS:
         * - En Ruta
         * - Novedad
         * - Devolucion
        */

        $domicilios = Domicilio::select(
            'domicilios.id as id',
            'domicilios.reference as reference',
            'domicilios.total as price',
            'domicilios.status as status',
            'domicilios.type as service',
            'domicilios.weight as weight',
            'domicilios.delivery as delivery',
            'domicilios.log_sender_id as sender_id',
            'domicilios.liquidated as liquidated',
            'domicilios.content as content',
            'clients.name as client',
            'cities.name as client_city',
            'zones.name as client_zone',
        )
            ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
            ->leftJoin('cities', 'cities.id', 'clients.city_id')
            ->leftJoin('zones', 'zones.id', 'clients.zone_id')
            ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.payment_method_id')
            ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.domicilio_id', 'domicilios.id')
            ->leftJoin('dispatch_relations', 'dispatch_relations.id', 'dispatch_relation_items.dispatch_relation_id')
            ->where(function ($query) {
                $query->where("domicilios.status", DomicilioStatusConst::ON_THE_WAY)
                    ->orWhere("domicilios.status", DomicilioStatusConst::NEW_EVENT)
                    ->orWhere("domicilios.status", DomicilioStatusConst::RETURNED);
            })
            ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
            //->whereBetween('domicilios.created_at', [$start, $end])
            ->where('dispatch_relations.id', $relationId)
            ->paginate($perPage);

        return $domicilios;
    }

    private function getDomiciliosByTabStatusWithoutAssing($perPage, $relationId)
    {
        /*
         * SE DEBE TRAER DOMICILIOS:
         * - En Ruta
         * - Novedad
         * - Devolucion
        */

        $domicilios = Domicilio::select(
            'domicilios.id as id',
            'domicilios.reference as reference',
            'domicilios.total as price',
            'domicilios.status as status',
            'domicilios.type as service',
            'domicilios.weight as weight',
            'domicilios.delivery as delivery',
            'domicilios.log_sender_id as sender_id',
            'domicilios.liquidated as liquidated',
            'domicilios.content as content',
            'clients.name as client',
            'cities.name as client_city',
            'zones.name as client_zone',
        )
            ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
            ->leftJoin('cities', 'cities.id', 'clients.city_id')
            ->leftJoin('zones', 'zones.id', 'clients.zone_id')
            ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.payment_method_id')
            ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.domicilio_id', 'domicilios.id')
            ->leftJoin('dispatch_relations', 'dispatch_relations.id', 'dispatch_relation_items.dispatch_relation_id')
            ->where("domicilios.status", DomicilioStatusConst::IN_WAREHOUSE)
            //  ->whereBetween('domicilios.created_at', [$start, $end])
            ->where('dispatch_relations.id', $relationId)
            ->where('domicilios.domiciliario_id', null)
            ->paginate($perPage);

        return $domicilios;
    }

    private function getDomiciliosCompleted($perPage, $domiciliarioId, $relationId)
    {
        /*
         * SE DEBE TRAER DOMICILIOS:
         * - Entregados
        */

        $domicilios = Domicilio::select(
            'domicilios.id as id',
            'domicilios.reference as reference',
            'domicilios.total as price',
            'domicilios.status as status',
            'domicilios.type as service',
            'domicilios.weight as weight',
            'domicilios.delivery as delivery',
            'domicilios.log_sender_id as sender_id',
            'domicilios.liquidated as liquidated',
            'domicilios.content as content',
            'clients.name as client',
            'cities.name as client_city',
            'zones.name as client_zone',
        )
            ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
            ->leftJoin('cities', 'cities.id', 'clients.city_id')
            ->leftJoin('zones', 'zones.id', 'zones.city_id')
            ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.payment_method_id')
            ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.domicilio_id', 'domicilios.id')
            ->leftJoin('dispatch_relations', 'dispatch_relations.id', 'dispatch_relation_items.dispatch_relation_id')
            ->where("domicilios.status", DomicilioStatusConst::DELIVERED)
            ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
            // ->whereBetween('domicilios.created_at', [$start, $end])
            ->where('dispatch_relations.id', $relationId)
            ->paginate($perPage);

        return $domicilios;
    }

    private function getAllDomiciliosByRelation($perPage, $domiciliarioId, $relationId)
    {
        $domicilios = Domicilio::select(
            'domicilios.id as id',
            'domicilios.reference as reference',
            'domicilios.total as price',
            'domicilios.status as status',
            'domicilios.type as service',
            'domicilios.weight as weight',
            'domicilios.delivery as delivery',
            'domicilios.log_sender_id as sender_id',
            'domicilios.liquidated as liquidated',
            'domicilios.content as content',
            'clients.name as client',
            'cities.name as client_city',
            'zones.name as client_zone',
        )
            ->leftJoin('clients', 'clients.id', 'domicilios.client_id')
            ->leftJoin('cities', 'cities.id', 'clients.city_id')
            ->leftJoin('zones', 'zones.id', 'zones.city_id')
            ->leftJoin('domicilio_collections', 'domicilio_collections.domicilio_id', 'domicilios.payment_method_id')
            ->leftJoin('dispatch_relation_items', 'dispatch_relation_items.domicilio_id', 'domicilios.id')
            ->leftJoin('dispatch_relations', 'dispatch_relations.id', 'dispatch_relation_items.dispatch_relation_id')
            ->where('dispatch_relations.domiciliario_id', $domiciliarioId)
            ->where("domicilios.status", "!=", DomicilioStatusConst::CANCELED)
            ->where('dispatch_relations.id', $relationId)
            ->distinct()
            ->paginate($perPage);

        return $domicilios;
    }

    /*public function generatePDF($id)
    {
        $relation = DispatchRelation::with(['dispatchRelationItems.domicilio'])->findOrFail($id);
        $configuration = \App\Configuration::all()->first();
        // Generar el PDF a partir de la vista
        $pdf = PDF::loadView('onx.relation', compact('relation', 'configuration'))->setPaper('a4', 'landscape');
        // Descargar el archivo PDF
        $path = public_path('pdf/');
        $fileName =  'relation_' . $relation->id . '.pdf';
        $pdf->save($path . '/' . $fileName);
        $file = public_path() . "/pdf/" . $fileName;
        if (ENV('APP_ENV') == 'production') {
            $url = url('pdf');
        } else {
            $url = url('pdf');
        }
        if (strpos(url('/'), 'ecomdex') !== false) {
            $url = url('api/public/pdf');
        }
        return "{$url}/{$fileName}";
    } */

    public function generatePDF($id)
    {
        // Obtener la relación con los domicilios y sus productos
        $relation = DispatchRelation::with(['dispatchRelationItems.domicilio'])->findOrFail($id);
        $configuration = \App\Configuration::all()->first();

        // Array para almacenar la cantidad total de cada producto por SKU
        $productQuantities = [];

        // Recorrer los domicilios de la relación
        foreach ($relation->dispatchRelationItems as $item) {
            $domicilio = $item->domicilio;

            // Decodificar el contenido del domicilio
            $content = json_decode($domicilio->content, true);

            if (is_array($content)) {
                foreach ($content as $product) {
                    if (isset($product['reference']) && isset($product['quantity'])) {
                        $sku = $product['reference'];
                        $quantity = (int)$product['quantity'];

                        // Sumar la cantidad al SKU correspondiente
                        if (!isset($productQuantities[$sku])) {
                            $productQuantities[$sku] = [
                                'name' => $product['name'] ?? 'Sin nombre', // Nombre del producto
                                'quantity' => 0,
                            ];
                        }
                        $productQuantities[$sku]['quantity'] += $quantity;
                    }
                }
            }
        }

        // Pasar el array de cantidades a la vista
        $pdf = PDF::loadView('onx.relation', compact('relation', 'configuration', 'productQuantities'))
            ->setPaper('a4', 'landscape');

        // Guardar el archivo PDF
        $path = public_path('pdf/');
        $fileName = 'relation_' . $relation->id . '.pdf';
        $pdf->save($path . '/' . $fileName);

        // Generar la URL del archivo
        $url = url('pdf');

        return "{$url}/{$fileName}";
    }

    public function generateRelationClose($relationId, $items)
    {
        $relation = DispatchRelation::with(
            [
                'dispatchRelationItems.domicilio',
                'dispatchRelationItems.domicilio.Collections'
            ]
        )
            ->findOrFail($relationId);

        $allItems = $relation->dispatchRelationItems->map->domicilio;

        $itemsDelivered = $relation->dispatchRelationItems->map->domicilio->filter(function ($item) {
            return $item && $item->status === DomicilioStatusConst::DELIVERED;
        });

        $itemsNotDelivered = $relation->dispatchRelationItems->map->domicilio->filter(function ($item) use ($itemsDelivered) {
            return $item && !$itemsDelivered->contains($item);
        });

        $paymentsSummary = $this->getPaymentsSummary($itemsDelivered);

        $productsDelivered = 0;
        $productsNotDelivered = 0;

        foreach ($itemsDelivered as $item) {
            $productsDelivered += $item->content_count;
        }

        foreach ($itemsNotDelivered as $item) {
            $productsNotDelivered += $item->content_count;
        }

        $this->domicilioService->setStatuses(
            $itemsDelivered->pluck('id')->toArray(),
            DomicilioStatusConst::VERIFIED
        );

        $configuration = Configuration::first();

        $pdf = Pdf::loadView('onx.relation_close', compact(
            'allItems',
            'relation',
            'configuration',
            'itemsDelivered',
            'itemsNotDelivered',
            'productsDelivered',
            'productsNotDelivered',
            'paymentsSummary'
        ))
            ->setPaper('a4', 'landscape');

        $fileName = 'relation_close' . $relation->id . '.pdf';
        $filePath = public_path('pdf/' . $fileName);

        $pdf->save($filePath);

        $baseUrl = url('pdf');

        if (strpos(url('/'), 'ecomdex') !== false) {
            $baseUrl = url('api/public/pdf');
        }

        return "{$baseUrl}/{$fileName}";
    }

    private function getPaymentsSummary($itemsDelivered)
    {
        return $itemsDelivered
            ->flatMap->Collections
            ->groupBy('payment')
            ->map(function ($collections, $paymentMethod) {
                return [
                    'payment_method' => $paymentMethod,
                    'total' => $collections->sum('value')
                ];
            })
            ->values();
    }
}
