<?php

namespace App\Services\Report;

use App\Configuration;
use App\Domiciliario;
use App\Domicilio;
use Illuminate\Support\Facades\DB;
use App\Traits\ApiResponser;
use stdClass;
use App\Enums\DomicilioStatusConst;
use Illuminate\Support\Facades\Mail;
use App\Mail\ReportDailyMail;
use App\Services\DomicilioCollection\DomicilioCollectionServices;

class ReportServices
{
    use ApiResponser;

    protected $domicilioCollectionService;

    public function __construct(DomicilioCollectionServices $domicilioCollectionService)
    {
        $this->domicilioCollectionService = $domicilioCollectionService;
    }

    /**
     * global report
     * @param int $page
     * @param int $perPage
     * @param string $from
     * @param string $to
     * @param string $search
     */

    public function globalReport($page, $perPage, $from, $to, $search, $status, $domiciliary, $liquidated, $city_id, $zone_id, $client_id, $sender_id, $payment_method)
    {
        try {
            // set columns
            $columns = [
                'id',
                'reference',
                'status',
                'collection',
                'date',
                'hour',
                'zone',
                'city',
                'client',
                'sender',
                'domiciliary',
            ];

            // get orders
            $orders = DB::table('domicilios')
                ->leftJoin('zones', 'zones.id', '=', 'domicilios.zone_id')
                ->leftJoin('cities', 'cities.id', '=', 'domicilios.city_id')
                ->leftJoin('clients', 'clients.id', '=', 'domicilios.client_id')
                ->leftJoin('log_senders', 'log_senders.id', '=', 'domicilios.log_sender_id')
                ->leftJoin('domiciliarios', 'domiciliarios.id', '=', 'domicilios.domiciliario_id')
                ->leftJoin('payment_methods', 'payment_methods.id', '=', 'domicilios.payment_method_id')
                ->leftJoin('statuses', function ($join) {
                    $join->on('statuses.domicilio_id', '=', 'domicilios.id')
                        ->whereIn('statuses.status', ['Entregado', 'Verificado']);
                })
                ->select(
                    'domicilios.id as id',
                    'domicilios.reference as reference',
                    'domicilios.status as status',
                    'domicilios.delivery as delivery',
                    'domicilios.liquidated as liquidated',
                    'domicilios.content as content',
                    'domicilios.domiciliario_id as domiciliario_id',
                    'domicilios.total as collection',
                    DB::raw("DATE_FORMAT(domicilios.created_at, '%Y/%m/%d') as date"),
                    DB::raw("DATE_FORMAT(domicilios.created_at, '%h:%i %p') as hour"),
                    DB::raw("DATE_FORMAT(domicilios.updated_at, '%Y/%m/%d') as updated_at"),
                    DB::raw("COALESCE(zones.name, 'No asignada') as zone"),
                    DB::raw("COALESCE(cities.name, 'No asignada') as city"),
                    DB::raw("COALESCE(clients.name, 'Sin cliente') as client"),
                    DB::raw("COALESCE(clients.phone, '') as client_phone"),
                    DB::raw("COALESCE(domiciliarios.name, 'No asignado') as domiciliary"),
                    DB::raw("COALESCE(payment_methods.description, 'No asignado') as payment"),
                    DB::raw("COALESCE(log_senders.name, 'Sin remitente') as sender"),
                    DB::raw("COALESCE(DATE_FORMAT(statuses.created_at, '%Y/%m/%d'), '') as delivery_date")
                )
                ->whereBetween('domicilios.created_at', [$from, $to])
                ->when(!is_null($search), function ($query) use ($search, $columns) {
                    $query->orHavingRaw("CONCAT_WS('', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
                })
                ->when($status != '', function ($query) use ($status) {
                    return $query->where('domicilios.status', $status);
                })
                ->when($domiciliary != '', function ($query) use ($domiciliary) {
                    return $query->where('domicilios.domiciliario_id', $domiciliary);
                })
                ->when($liquidated != '', function ($query) use ($liquidated) {
                    return $query->where('domicilios.liquidated', $liquidated);
                })
                ->when($city_id != '', function ($query) use ($city_id) {
                    return $query->where('domicilios.city_id', $city_id);
                })
                ->when($zone_id != '', function ($query) use ($zone_id) {
                    return $query->where('domicilios.zone_id', $zone_id);
                })
                ->when($client_id != '', function ($query) use ($client_id) {
                    return $query->where('domicilios.client_id', $client_id);
                })
                ->when($sender_id != '', function ($query) use ($sender_id) {
                    return $query->where('domicilios.log_sender_id', $sender_id);
                })
                ->when($payment_method != '', function ($query) use ($payment_method) {
                    return $query->where('domicilios.payment_method_id', $payment_method);
                })
                ->whereNull('domicilios.deleted_at')
                ->groupBy('domicilios.id')
                ->paginate($perPage);

            // return data to controllers
            return [
                'orders' => $orders->items(),
                'totalPage' => $orders->total(),
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    public function getMonthReport($from, $to)
    {
        $result = new stdClass();

        $result->total_orders = Domicilio::whereBetween('created_at', [$from, $to])
            ->count();

        $result->delivered_success = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::DELIVERED)
            ->count();

        $result->orders_in_warehouse = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::IN_WAREHOUSE)
            ->count();

        $result->orders_on_the_way = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::ON_THE_WAY)
            ->count();

        $result->orders_pending = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::PENDING)
            ->count();

        $result->orders_canceled = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::CANCELED)
            ->count();

        $result->orders_new = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::NEW_EVENT)
            ->count();

        $result->orders_rescheduled = Domicilio::whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::RESCHEDULED)
            ->count();

        return $result;
    }

    public function getTodayReport()
    {
        $result = new stdClass();
        $configuration = Configuration::first();

        $currentDate = date('Y-m-d');

        $result->total_orders = Domicilio::whereDate('created_at', '=', $currentDate)
            ->count();

        $result->delivered_success = Domicilio::whereDate('updated_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::DELIVERED)
            ->count();

        $result->delivered_success_same_day = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::DELIVERED)
            ->count();

        $result->orders_in_warehouse = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::IN_WAREHOUSE)
            ->count();

        $result->orders_on_the_way = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::ON_THE_WAY)
            ->count();

        $result->orders_pending = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::PENDING)
            ->count();

        $result->orders_canceled = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::CANCELED)
            ->count();

        $result->orders_new = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::NEW_EVENT)
            ->count();

        $result->orders_rescheduled = Domicilio::whereDate('created_at', '=', $currentDate)
            ->where('status', DomicilioStatusConst::RESCHEDULED)
            ->count();

        $totalCollectionDaily = $this->domicilioCollectionService->getTotalCollectionByDate();

        $result->totalCollection = "$totalCollectionDaily {$configuration->getCurrencySimbolAttribute()}";

        return $result;
    }

    public function getMonthReportByDomiciliario($domiciliarioId, $from, $to)
    {
        $domiciliario = Domiciliario::findOrFail($domiciliarioId);
        $result = new stdClass();
        $result->domiciliary_name = $domiciliario->name;
        $result->total_orders = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->count();

        $result->delivered_success = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::DELIVERED)
            ->count();

        $result->orders_in_warehouse = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::IN_WAREHOUSE)
            ->count();

        $result->orders_on_the_way = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::ON_THE_WAY)
            ->count();

        $result->orders_pending = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::PENDING)
            ->count();

        $result->orders_canceled = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::CANCELED)
            ->count();

        $result->orders_new = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::NEW_EVENT)
            ->count();

        $result->orders_rescheduled = Domicilio::where("domiciliario_id", $domiciliarioId)
            ->whereBetween('created_at', [$from, $to])
            ->where('status', DomicilioStatusConst::RESCHEDULED)
            ->count();

        return $result;
    }

    public function sendDailyReport()
    {
        // Obtener el reporte del día actual
        $report = $this->getTodayReport();

        // Datos para el correo
        $data = [
            'total_orders' => $report->total_orders,
            'delivered_success' => $report->delivered_success,
            'delivered_success_same_day' => $report->delivered_success_same_day,
            'orders_in_warehouse' => $report->orders_in_warehouse,
            'orders_on_the_way' => $report->orders_on_the_way,
            'orders_pending' => $report->orders_pending,
            'orders_canceled' => $report->orders_canceled,
            'orders_new' => $report->orders_new,
            'orders_rescheduled' => $report->orders_rescheduled,
            'total_collection' => $report->totalCollection
        ];

        /*
         CODIGO PARA ENVIAR EMAIL A VARIOS DESTINATARIOS
        $recipients = [
            'recipient1@example.com',
            'recipient2@example.com',
            'recipient3@example.com',
        ];

        // Enviar correo
        Mail::to($recipients)->send(new ReportDailyMail($data));
        */

        // Enviar correo
        Mail::to('juan@onexmedia.co')->send(new ReportDailyMail($data));

        return response()->json(['message' => 'Reporte enviado exitosamente.']);
    }
}
