<?php

namespace App\Services\Rescheduling;

use App\Domicilio;
use App\DomicilioNews;
use App\DomiciliosRescheduling;
use App\Services\Domicilio\DomicilioServices;
use Illuminate\Support\Facades\Auth;
use App\Status;
use App\Traits\ApiResponser;
use App\Enums\DomicilioStatusConst;
use App\Enums\TypeDomicilioSolutionConst;
use App\Services\Onexfy\OnexfyService;

class ReschedulingService
{
    use ApiResponser;

    protected $domicilioService;
    protected $onexfyService;

    function __construct(DomicilioServices $domicilioService, OnexfyService $onexfyService)
    {
        $this->domicilioService = $domicilioService;
        $this->onexfyService = $onexfyService;
    }

    public function setStatuses(
        $domicilioId,
        $statusCode,
        $description,
        $date,
        $startTime,
        $endTime,
        $file = null
    ) {
        $rescheduling = DomiciliosRescheduling::create([
            "domicilio_id" => $domicilioId,
            "rescheduling_date" => $date,
            "rescheduling_start_hour" => $startTime,
            "rescheduling_end_hour" => $endTime,
            "description" => $description,
            "modified_by" => Auth::id()
        ]);

        $domicilio = Domicilio::findOrFail($domicilioId);
        $domicilio->status = DomicilioStatusConst::RESCHEDULED;
        $domicilio->save();

        Status::create([
            'status' => $statusCode,
            'observation' => $description,
            'domicilio_id' => $domicilioId,
            'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
            'updated_by' => Auth::id(),
        ]);

        if (isset($file)) {
            $domicilioCollectionFile = $this->domicilioService->uploadEvidence(
                $file,
                $domicilioId,
                $statusCode
            );

            $rescheduling->evidence_id = $domicilioCollectionFile->id;
            $rescheduling->save();
        }

        return $rescheduling;
    }

    public function solveOrderWithRescheduling($reference, $reschedulingBody)
    {
        $domicilio = Domicilio::where("reference", $reference)->get()->first();

        if (!isset($domicilio)) {
            throw new \Exception("No existe el domicilio nro de guia {$reference}", 422);
        }

        if ($domicilio->status === DomicilioStatusConst::RESCHEDULED) {
            return $domicilio;
        }

        $reschedule = $this->setStatuses(
            $domicilio->id,
            DomicilioStatusConst::RESCHEDULED,
            $reschedulingBody["description"],
            $reschedulingBody["date"],
            $reschedulingBody["startTime"],
            $reschedulingBody["endTime"],
            $reschedulingBody["file"] ?? null,
        );

        if (!$reschedule) {
            throw new \Exception("Ha ocurrido un error reprogramando la guia {$reference}", 422);
        }

        $this->onexfyService->notifyChangeStatusOrder(
            $domicilio->reference,
            DomicilioStatusConst::RESCHEDULED,
            "reprogramacion"
        );

        return $domicilio;
    }

    public function solveDomicilio($reference, $body)
    {
        switch ($body["type_solution"]) {
            case TypeDomicilioSolutionConst::RESCHEDULING:
                $params = $body["data"];
                return $this->solveOrderWithRescheduling($reference, $params);
            case TypeDomicilioSolutionConst::CANCELLED:
                return $this->domicilioService->cancelOrder($reference);
        }
    }
}
