<?php

namespace App\Services\User;

use App\User;
use App\Traits\ApiResponser;
use App\WalletCharge;
use Illuminate\Support\Str;
use App\Http\Controllers\AuditController;
use App\OrderGuideCommission;
use App\OrderGuide;
use Illuminate\Support\Facades\DB;
use App\CoordinatorHasCity;
use App\Enums\RolesUserConst;
use Illuminate\Support\Facades\Log;

class UserServices
{
    use ApiResponser;

    public function __construct() {}

    /**
     * Create user
     * @param array $params
     */
    public function createUser(array $params)
    {
        try {

            if ($params["role_id"] == RolesUserConst::REGIONAL_COORDINATOR) {

                if (!isset($params["cities"]) || count($params["cities"]) === 0) {
                    return $this->handlerException('Debe seleccionar las ciudades que el coordinador va a administrar');
                }
            }

            $isset = User::where('email', $params['email'])->first();
            if (isset($isset)) {
                return $this->handlerException('Existe un usuario con este correo electrónico');
            }
            $user = User::create($params);

            if (isset($params["cities"]) && $params["role_id"] == RolesUserConst::REGIONAL_COORDINATOR) {
                $cities = $params["cities"];
                foreach ($cities as $city) {
                    CoordinatorHasCity::create([
                        "user_id" => $user->id,
                        "city_id" => $city
                    ]);
                }
            }
            return $user;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * update sender user
     * @param int $senderId
     */
    public function updateUserSender($senderId, $params)
    {
        try {
            $user = User::where('log_sender_id', $senderId)->first();
            if ($user) {
                $user->update([
                    'email' => $params['email'],
                    'name' => $params['email']
                ]);
                if (isset($params['password']) and !is_null($params['password'])) {
                    $user->update([
                        'password' => bcrypt($params['password']),
                    ]);
                }
            }
            return $user;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * List user
     * @param int $perPage
     * @param string $search
     */

    public function listUsers(int $perPage, string $search)
    {
        try {
            $idRegionalCoordinator = RolesUserConst::REGIONAL_COORDINATOR;

            $columns = [
                'id',
                'name',
                'email',
                'role',
                'role_id'
            ];
            $users = DB::table('users')
                ->leftJoin('roles', 'roles.id', '=', 'users.role_id')
                ->select(
                    'users.id as id',
                    'users.name as name',
                    'users.email as email',
                    'roles.name as role',
                    'roles.id as role_id',
                    DB::raw("
                         CASE
                             WHEN users.role_id = {$idRegionalCoordinator} THEN (
                                 SELECT JSON_ARRAYAGG(cities.id)
                                 FROM coordinator_has_cities
                                 JOIN cities ON cities.id = coordinator_has_cities.city_id
                                 WHERE coordinator_has_cities.user_id = users.id
                             )
                             ELSE NULL
                         END as cities
                     ")
                )
                ->when(!is_null($search), function ($query) use ($search, $columns) {
                    return $query->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
                })
                ->whereNull('users.deleted_at')
                ->whereNotIn('role_id', [RolesUserConst::REMITTENT, RolesUserConst::DOMICILIARIO])
                ->paginate($perPage);

            // Convertir el campo cities a un arreglo de IDs
            $users->getCollection()->transform(function ($user) {
                if (!is_null($user->cities)) {
                    $user->cities = json_decode($user->cities, true);
                }
                return $user;
            });

            return $users;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }


    /**
     * filter user
     * @param int $id
     */
    public function filterById(int $id)
    {
        try {
            $users = DB::table('users')
                ->leftJoin('roles', 'roles.id', 'users.role_id')
                ->select(
                    'users.id as id',
                    'users.name as name',
                    'users.email as email',
                    'roles.name as role',
                    'roles.id as role_id'
                )
                ->where('users.id', $id)
                ->whereNotIn('role_id', [RolesUserConst::DOMICILIARIO, RolesUserConst::REMITTENT])
                ->whereNull('users.deleted_at')
                ->first();
            return $users;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Delete user
     * @param User $user
     */
    public function deleteUser(User $user)
    {
        try {
            $user->delete();
            return $user;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Create user
     * @param array $params
     */
    public function updateUser(User $user, array $params)
    {
        try {
            $user->update($params);

            if ($params["role_id"] === RolesUserConst::REGIONAL_COORDINATOR) {
                $this->updateCitiesAdministration($params["cities"], $user);
            }
            return $this->filterById($user->id);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    private function updateCitiesAdministration($cities, User $user)
    {
        //SE REINICIA LAS CIUDADES ADMINSITRADAS POR EL USUARIO
        CoordinatorHasCity::where("user_id", $user->id)->delete();

        foreach ($cities as $city) {
            CoordinatorHasCity::create([
                "user_id" => $user->id,
                "city_id" => $city
            ]);
        }
    }
}
