<?php

namespace App\Services\Zone;

use Illuminate\Support\Facades\DB;
use App\Zone;
use App\Traits\ApiResponser;
use App\Exports\ZonesExport;
use App\LogSender;
use Maatwebsite\Excel\Facades\Excel;

final class ZoneServices
{
    use ApiResponser;

    /**
     * create new zone
     * @param array $params
     */
    public function storeZone(array $params)
    {
        try {
            $zone = Zone::create($params);
            return $zone;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List zones
     * @param int $city
     * @param int $perPage
     * @param int $page
     */
    public function listZones(int $city, int $page, int $perPage, string $search)
    {
        try {
            // columsn
            $columns = [
                'id',
                'name',
                'code',
                'abbreviation',
                'city',
                'value',
                'finally_price'
            ];
            // do query
            $zone = DB::table('zones')
                ->leftJoin('cities', 'cities.id', 'zones.city_id')
                ->select(
                    'zones.id as id',
                    'zones.name as name',
                    'zones.code as code',
                    'zones.value as value',
                    'cities.abbreviation as abbreviation',
                    'cities.name as city',
                    'zones.finally_price as finally_price'
                )
                ->where('zones.city_id', $city)
                ->when(!is_null($search), function ($query) use ($search, $columns) {
                    return $query->orHavingRaw("CONCAT_WS(' ', " . implode(', ', $columns) . ") LIKE '%" . $search . "%'");
                })
                ->paginate($perPage);
            // return
            return [
                'items' => $zone->items(),
                'totalItems' => $zone->total(),
                'page' => $page,
            ];
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * delete zone
     * @param Zone $zone
     */
    public function deleteZone(Zone $zone)
    {
        try {
            $zone->delete();
            return $zone;
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * downlaod Excell
     */
    public function exportExcel()
    {
        try {
            return Excel::download(new ZonesExport, 'zones.xlsx');
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List by cities
     * @param int $city
     * @param int $page
     * @param int $perPage
     */
    public function listZonesByCity($city, $page, $perPage)
    {
        try {
            $zones = DB::table('zones')
                ->leftJoin('cities', 'cities.id', 'zones.city_id')
                ->select(
                    'zones.id as id',
                    'zones.name as name',
                    'zones.code as code',
                    'cities.name as city',
                    'cities.abbreviation as abbreviation',
                    'zones.finally_price as order_price'
                )
                ->where('zones.city_id', $city)
                ->paginate($perPage);
            // return data
            return $this->successResponse([
                'success' => true,
                'message' => 'Listado de zonas.',
                'data' => $zones->items(),
                'totalItems' => $zones->total(),
                'page' => $page,
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }

    /**
     * List by cities with log sender (remitente)
     * @param int $city
     * @param int $page
     * @param int $perPage
     */
    public function listZonesByCityWithLogSender($city, $logSenderId, $page, $perPage)
    {
        try {
            // Obtener el logSender y su descuento
            $logSender = LogSender::where("onexfy_id", $logSenderId)->first();
            $discount = $logSender->discount ?? 0;

            $zones = DB::table('zones')
                ->leftJoin('cities', 'cities.id', '=', 'zones.city_id')
                ->select(
                    'zones.id as id',
                    'zones.name as name',
                    'zones.code as code',
                    'cities.name as city',
                    'cities.abbreviation as abbreviation',
                    DB::raw("ROUND(zones.finally_price - (zones.finally_price * {$discount} / 100), 2) as order_price")
                )
                ->where('zones.city_id', $city)
                ->paginate($perPage);

            // Formatear order_price a 2 decimales
            if ($logSender) {
                $zones->transform(function ($zone) use ($logSender) {
                    $zone->order_price = number_format((float)$zone->order_price, 2, '.', '');
                    $zone->fullfilment = $logSender->fullfilment;
                    $zone->discount = $logSender->discount;
                    $zone->tax_collection = $logSender->tax_collection;
                    return $zone;
                });
            } else {
                $zones->transform(function ($zone) use ($logSender) {
                    $zone->order_price = number_format((float)$zone->order_price, 2, '.', '');
                    return $zone;
                });
            }

            // Retornar datos
            return $this->successResponse([
                'success' => true,
                'message' => 'Listado de zonas.',
                'data' => $zones->items(),
                'totalItems' => $zones->total(),
                'page' => $page,
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }


    /**
     * update zones
     * @param Zone $zone
     * @param array $params
     */
    public function updateZone(Zone $zone, $params)
    {
        try {
            $zone->update($params);
            // return data
            return $this->successResponse([
                'success' => true,
                'message' => 'Se ha modificado la zona correctamente.',
                'data' => $zone->refresh()
            ]);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage(), 400);
        }
    }
}
