<?php

namespace App\Imports;

use DB;
use Illuminate\Support\Str;
use App\Traits\ApiResponser;
use Illuminate\Support\Collection;
use App\Http\Controllers\ClientController;
use Maatwebsite\Excel\Concerns\ToCollection;

class DomicilioImport implements ToCollection
{
    use ApiResponser;

    public $orders, $items;

    /**
     * constructor
     */
    public function __construct() {
        $this->orders = [];
        $this->items = [];
    }

    /**
    * @param Collection $collection
    */
    public function collection(Collection $rows)
    {
        //
        DB::beginTransaction();
        try {
            foreach ($rows as $key => $row) {
                if (!is_null($row[0]) and $row[0] !== 'Pos') {
                    // si es una orden
                    if ($row[1] == 'Domicilio') {
                        // validaciones
                        $this->valdiateEmptyData($row, $key);
                        // prepare order data
                        $this->prepareOrderData($row, $key);
                    } else {
                        $this->prepareItemsData($row, $key);
                    }
                }
            }
            $this->saveDataOnBbDd();
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * Prepare order data
     * @param $row
     */
    public function prepareOrderData($row, $line) {
        try {
            // valdiamos el remitente y el domiciliario
            $sender = \App\LogSender::where('dni', $row[2])->first();
            if (is_null($sender)) {
                return $this->handlerException("El remitente en la fila {$line} no existe.");
            }
            $domiciliary = \App\Domiciliario::where('dni', $row[3])->first();
            if (is_null($domiciliary)) {
                return $this->handlerException("El domiciliario en la fila {$line} no existe.");
            }
            $payment = DB::table('payment_methods')->where('description', $row[6])->first();
            if (is_null($payment)) {
                return $this->handlerException("El medio de pago en la fila {$line} no existe.");
            }
            $city = \App\City::where('abbreviation', $row[10])->first();
            if (is_null($city)) {
                return $this->handlerException("La ciudad en la fila {$line} no existe.");
            }
            $zone = \App\Zone::where('code', $row[11])->first();
            if (is_null($zone)) {
                return $this->handlerException("La zona en la fila {$line} no existe.");
            }
            // prepare order data
            $order = [
                'pos' => $row[0],
                'delivery' => $row[4],
                'domiciliario_id' => $domiciliary->id ? $domiciliary->id : null,
                'log_sender_id' => $sender ? $sender->id : null,
                'observation' => !is_null($row[13]) ? $row[13] : '',
                'type' => 'Express',
                'status' => 'Pendiente',
                'total' => !is_null($row[5]) ? $row[5] : 0,
                'payment_method_id' => $payment ? $payment->id : null,
                'city_id' => $city ? $city->id : null,
                'zone_id' => $zone ? $zone->id : null,
                'client' => [
                    'address' => $row[9],
                    'city_id' => $city ? $city->id : null,
                    'zone_id' => $zone ? $zone->id : null,
                    'name' => $row[7],
                    'dni' => $row[8],
                    'phone' => round($row[12], 0),
                ]
            ];
            array_push($this->orders, $order);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * prepare items data
     */
    public function prepareItemsData($row, $line) {
        try {
            // som validations
            if (is_null($row[14])) {
                return $this->handlerException("Debes especificar el nombre del producto que contiene el paquete para la fila: {$line}");
            }
            if (is_null($row[16])) {
                return $this->handlerException("Debes especificar la cantidad de productos que contiene el paquete para la fila: {$line}");
            }
            // prepare items
            $item = [
                'name' => $row[14],
                'quantity' => $row[16],
                'price' => !is_null($row[17]) ? $row[17] : null,
                'sku' => !is_null($row[15]) ? $row[15] : null,
                'pos' => $row[0],
            ];
            array_push($this->items, $item);
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }

    /**
     * validate emty data
     * @param $row
     */
    public function valdiateEmptyData($row, $line) {
        if (is_null($row[0])) {
            return $this->handlerException("Debes especificar la Pos en el excel para la fila: {$line}");
        }
        if (is_null($row[1])) {
            return $this->handlerException("Debes especificar el tipo de linea para la fila {$line}");
        }
        if (is_null($row[2])) {
            return $this->handlerException("Debes especificar el remitente en la fila {$line}");
        }
        if (is_null($row[3])) {
            return $this->handlerException("Debes especificar el domiciliario en la fila {$line}");
        }
        if (is_null($row[4])) {
            return $this->handlerException("Debes especificar el costo del domicilio en la fila {$line}");
        }
        if (is_null($row[5])) {
            return $this->handlerException("Debes especificar valor a recaudar en la fila {$line}");
        }
        if (is_null($row[6])) {
            return $this->handlerException("Debes especificar el método de pago en la fila {$line}");
        }
        if (is_null($row[7])) {
            return $this->handlerException("Debes especificar el destinatario en la fila {$line}");
        }
        if (is_null($row[8])) {
            return $this->handlerException("Debes especificar el documento de el destinatario en la fila {$line}");
        }
        if (is_null($row[9])) {
            return $this->handlerException("Debes especificar la dirección de el destinatario en la fila {$line}");
        }
        if (is_null($row[10])) {
            return $this->handlerException("Debes especificar la ciudad de el destinatario en la fila {$line}");
        }
        if (is_null($row[11])) {
            return $this->handlerException("Debes especificar la zona de el destinatario en la fila {$line}");
        }
        if (is_null($row[12])) {
            return $this->handlerException("Debes especificar el teléfono de el destinatario en la fila {$line}");
        }
    }

    /**
     * save order in bbdd
     */
    public function saveDataOnBbDd() {
        try {
            foreach ($this->orders as $ord => $order) {
                // valdiamos y creamos el cliente
                $client = $order['client'];
                $clientBd = \App\Client::where('dni', $client['dni'])->first();
                if (!$clientBd) {
                    $clientBd = \App\Client::create($order['client']);
                }
                // unset data
                unset($order['client']);
                // creamos el domicilio
                $order['client_id'] = $clientBd ? $clientBd->id : null;
                $domicilio = \App\Domicilio::create($order);
                $reference = Str::padLeft($domicilio->id, 9, 0);
                $domicilio->update(['reference' => $reference]);
                // generate initial status
                $status = \App\Status::create([
                    'status' => 'Pendiente',
                    'observation' => isset($order['observation']) ? $order['observation'] : '',
                    'domicilio_id' => $domicilio->id,
                    'domiciliario' => !is_null($domicilio->domiciliario) ? $domicilio->domiciliario->name : '',
                ]);
                // validamos los productos
                $pos = $order['pos'];
                $filteredProducts = array_filter($this->items, function ($item) use ($pos) {
                    return isset($item['pos']) && $item['pos'] == $pos;
                });
                if (count($filteredProducts) > 0) {
                    foreach ($filteredProducts as $prod => $product) {
                        \App\DomicilioProduct::create([
                            'reference' => isset($product['sku']) ? $product['sku'] : '',
                            'name' => isset($product['name']) ? $product['name'] : '',
                            'price' => isset($product['price']) ? $product['price'] : '',
                            'quantity' => isset($product['quantity']) ? $product['quantity'] : '',
                            'domicilio_id' => $domicilio->id
                        ]);
                    }
                }
            }
        } catch (\Exception $e) {
            return $this->handlerException($e->getMessage());
        }
    }
}
