<?php

namespace App\Services\Domiciliario;

use App\Domiciliario;
use App\User;
use App\Helpers\Helper;
use App\Http\Requests\Domiciliario\PostRequest;
use App\Http\Controllers\AuditController;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\AuthController;
use App\DomiciliarioHasCities;

class DomiciliarioService
{

    public function createDomiciliario(PostRequest $request)
    {
        $mainId = $request->user()->getIdMain($request->user());

        // Verificar si el email o el DNI ya están en uso
        $this->checkIfExists($request->email, 'email');
        $this->checkIfExists($request->dni, 'dni');

        // Subir imagen
        $path = $this->uploadProfilePhoto($request->file);
        $payload = $request->all();
        $password = Hash::make($request['password']);

        $domiciliario = Domiciliario::create([
            'name' => $payload["name"],
            'email' => $payload["email"],
            'dni' => $payload["dni"],
            'phone' => $payload["phone"],
            'placa' => $payload["plate"],
            'photo' => $path,
            'password' => $password,
            'username' => $payload["dni"],
            'hiring_method' => $payload["hiring_method"]
        ]);

        // Registrar usuario
        $user = User::create([
            'name' => $payload["name"],
            'email' => $payload["email"],
            'username' => $payload["dni"],
            'password' => $password,
            'domiciliario_id' => $domiciliario->id
        ]);

        $city = $payload["city"];
        DomiciliarioHasCities::create([
            "domiciliario_id" => $domiciliario->id,
            "city_id" => $city
        ]);
        $domiciliario->hiring_method = $request['hiring_method'];


        // Auditoría
        $this->logAudit($request->user()->name, $domiciliario->name, $mainId);

        return $domiciliario;
    }

    public function updateDomiciliario($request, Domiciliario $domiciliario)
    {
        // Procesar cambios en la foto de perfil
        if ($request->has('file')) {
            $domiciliario->photo = $this->uploadProfilePhoto($request->file);
        }

        // Actualizar contraseña si está presente
        if ($request->has('password')) {
            $request['password'] = bcrypt($request['password']);
        }

        // Actualizar placa si está presente
        if ($request->has('plate')) {
            $domiciliario->placa = $request->plate;
        }

        // Actualizar los datos del domiciliario
        $domiciliario->update($request->all());

        // Buscar el usuario relacionado
        $user = User::where('domiciliario_id', $domiciliario->id)->first();

        // Actualizar o registrar usuario relacionado
        if ($user) {
            $user->update([
                'email' => $domiciliario->email,
                'username' => $domiciliario->dni,
            ]);

            if ($request->has('password')) {
                $user->update(['password' => $request['password']]);
            }
        } else {
            $user = AuthController::register(
                $domiciliario->email,
                $domiciliario->dni,
                $domiciliario->password,
                $domiciliario->id
            );
        }

        if ($request->city) {
            $domiciliarioHasCity = DomiciliarioHasCities::where("domiciliario_id", $domiciliario->id)->first();
            if ($domiciliarioHasCity) {
                $domiciliarioHasCity->update([
                    "city_id" => $request->city
                ]);
            } else {
                DomiciliarioHasCities::create([
                    "domiciliario_id" => $domiciliario->id,
                    "city_id" => $request->city
                ]);
            }
        }

        // Retornar el domiciliario actualizado
        return $domiciliario;
    }


    protected function checkIfExists($value, $field)
    {
        $exists = Domiciliario::where($field, $value)->exists();
        if ($exists) {
            throw new \Exception("Ya existe un domiciliario con este {$field}");
        }
    }

    protected function logAudit($userName, $domiciliarioName, $mainId)
    {
        // Lógica para registrar en auditoría
        AuditController::store($userName, 'Ha creado al domiciliario: ' . $domiciliarioName, 'Domiciliario', $mainId);
    }

    private function uploadProfilePhoto($file)
    {
        if (isset($file)) {
            $path = Helper::uploadImage($file, 'domiciliario');
            $urlPath = 'images/domiciliario';
            if (strpos(url('/'), 'ecomdex') !== false) {
                $urlPath = 'api/public/images/domiciliario';
            }
            return url($urlPath, $path);
        }
    }
}
