<?php

namespace App\Services\Onexfy;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\Onexfy\StatusHandlers\StatusHandlerFactory;
use App\Services\Onexfy\StatusHandlers\StatusUpdateDTO;

class OnexfyService
{
    protected $apiUrl;
    protected $prefix;
    protected $statusHandlerFactory;

    public function __construct(StatusHandlerFactory $statusHandlerFactory)
    {
        $this->statusHandlerFactory = $statusHandlerFactory;
        $this->apiUrl = config('services.onexfy.api_url');
        $this->prefix = env("ONEX_DELIVERY_PREFIX");
    }

    public function notifyChangeStatusOrder(
        string $reference,
        string $status,
        string $note,
        ?string $content = null
        //SI SE NECESITA OTRO PARAMETRO, SE AGREGA
    ) {
        try {

            $request = new StatusUpdateDTO(
                $reference,
                $note,
                $status,
                $content
            );

            $handler = $this->statusHandlerFactory->getHandler($status);
            $params = $handler->handle($request);

            Log::info('Onexfy API Request', [
                'endpoint' => "{$this->apiUrl}/api/v2/update-order-status",
                'params' => $params,
            ]);

            $response = $this->makeApiRequest($params);

            if ($response->successful()) {
                Log::info('Onexfy API Successful Response', [
                    'endpoint' => "{$this->apiUrl}/api/v2/update-order-status",
                    'response' => $response->json(),
                ]);
                return $response->json();
            }

            return $this->errorResponse($response->body());
        } catch (\Exception $e) {
            Log::critical('Onexfy API Exception', [
                'endpoint' => "{$this->apiUrl}/api/v2/update-order-status",
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return $this->errorResponse($e->getMessage());
        }
    }

    protected function makeApiRequest(array $params)
    {
        return Http::withHeaders([
            'Custom-Auth-Token' => env("CUSTOM_AUTH_TOKEN"),
        ])->post("{$this->apiUrl}/api/v2/update-order-status", $params);
    }

    protected function errorResponse(string $message)
    {
        return [
            'error' => true,
            'message' => $message,
        ];
    }
}

/*
namespace App\Services\Onexfy;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\Domicilio\DomicilioSettlementService;
use App\Enums\DomicilioStatusConst;

class OnexfyService
{
    protected $apiUrl;
    protected $prefix;
    protected $domicilioSettlementService;

    public function __construct(DomicilioSettlementService $domicilioSettlementService)
    {
        $this->domicilioSettlementService = $domicilioSettlementService;
        $this->apiUrl = config('services.onexfy.api_url');
        $this->prefix = env("ONEX_DELIVERY_PREFIX");
    }

    public function notifyChangeStatusOrder($reference, $status, $note)
    {
        $params = [];

        try {
            $settlementData = null;

            if (DomicilioStatusConst::VERIFIED === $status) {
                $settlementData = $this->domicilioSettlementService->calculateSettlementByDomicilio($reference);
            }

            if (is_object($settlementData)) {
                if (!$this->domicilioSettlementService->isThereDropshipper($reference) && DomicilioStatusConst::VERIFIED == $status) {
                    $settlementData->provider_settlement = number_format($settlementData->total, 2);
                }

                $params = [
                    "reference" => $reference,
                    "status" => $status,
                    "note" => $note,
                    "dropshipper_settlement" => $settlementData->dropshipper_settlement ?? 0,
                    "prov_settlement" => $settlementData->provider_settlement ? number_format($settlementData->provider_settlement, 2) : 0
                ];
            } else {
                $params = [
                    "reference" => $reference,
                    "status" => $status,
                    "note" => $note,
                    "dropshipper_settlement" => 0,
                    "prov_settlement" => 0
                ];
            }

            if ($status == DomicilioStatusConst::RETURNED) {
                $params['content'] = true;
            }

            // Verificar los datos que estás enviando antes de la solicitud
            Log::info('Onexfy API Request', [
                'endpoint' => "{$this->apiUrl}/api/v2/update-order-status",
                'params' => $params,
            ]);

            $response = Http::withHeaders([
                'Custom-Auth-Token' => env("CUSTOM_AUTH_TOKEN"), // Asegúrate de que esté cargando correctamente el token
            ])
                ->post("{$this->apiUrl}/api/v2/update-order-status", $params);

            if ($response->successful()) {
                Log::info('Onexfy API Successful Response', [
                    'endpoint' => "{$this->apiUrl}/api/v2/update-order-status",
                    'params' => $params,
                    'response' => $response->json(),
                ]);

                return $response->json();
            }

            return [
                'error' => true,
                'message' => $response->body(),
            ];
        } catch (\Exception $e) {
            Log::critical('Onexfy API Exception', [
                'endpoint' => "{$this->apiUrl}/api/v2/update-order-status",
                'params' => $params,
                'exception' => $e->getMessage(),
            ]);

            return [
                'error' => true,
                'message' => $e->getMessage(),
            ];
        }
    }
}
*/